#/usr/bin/env python
# encoding: utf-8

"""Babglet provides basic simplifications for using pyglet in games. 

classes: 
    - Scene: A window with an event_loop. To use it, you must call the self.event_loop() in the init of the main subclass, and you must implement self.action()
    - Thing: A thing, the basic movable element. It has a position (x and y), an image, can move and detect if it touches an other Thing, and it can be told to move away fuzzily when it touches an other thing. 


Call them into your project via: 

import babglet
class Scene(babglet.Scene):: 
    def __init__(self): 
        super(Scene, self).init()
        # You should add a list of actors
        self.actors = [thing1, thing2, thing3]
        # Your own Scene attribute and calls which should be done before the event loop (which means before the game ends)
        
        # Start the event_loop. 
        self.event_loop()
        
    def action(self):: 
        for i in self.actors: 
            i.act()
            i.move()
            i.blit()


doctests of touch(thing, thang), (width: 10, height: 11): 
    >>> # both at the same place
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    True
    >>> # thang right of thing
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=10,  y=0)
    >>> thing.touches(thang)
    True
    >>> # thang left of thing
    >>> thing = Thing(x=10,  y=0) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    True
    >>> # thang below thing
    >>> thing = Thing(x=0,  y=11) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    True
    >>> # thang above thing
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=0,  y=11)
    >>> thing.touches(thang)
    True
    >>> # thing upper right corner of thang
    >>> thing = Thing(x=10,  y=11) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    True
    >>> # thang upper right corner of thing
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=10,  y=11)
    >>> thing.touches(thang)
    True
    >>> # thang far above thing
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=0,  y=12)
    >>> thing.touches(thang)
    False
    >>> # thang far right of thing
    >>> thing = Thing(x=0,  y=0) 
    >>> thang = Thing(x=11,  y=0)
    >>> thing.touches(thang)
    False
    >>> # thing far above thang
    >>> thing = Thing(x=0,  y=12) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    False
    >>> # thing far right of thang
    >>> thing = Thing(x=11,  y=0) 
    >>> thang = Thing(x=0,  y=0)
    >>> thing.touches(thang)
    False
"""

# First we need PyGlet,  window and image for the dot

from pyglet import window,  image #,  media

# And Keyboard events

from pyglet.window import key

# For the alpha Channel we also need pyglet.gl

from pyglet.gl import glEnable,  glBlendFunc,  GL_BLEND,  GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA

# And for a fix frame rate, we need to be able to let the pyglet.clock tick

from pyglet.clock import tick as clock_tick

# Also we need path from os, to be able to call files with platform independent paths. 

from os import path

# And random movement for collision avoidance

from random import randrange

# And we want to have a maximum frame rate, so we need to be able to make ourselves sleep. 

from time import sleep


# Now we first create the scene class as a general class to write easy games. 

class Scene(window.Window): 
    """The base class for any game. 
    
    Init the screen and an event loop which evokes the function self.action() for any action of the game."""
    def __init__(self,  scene_name="Default Scene",  width=1200,  height=800,  fullscreen=False,  resizable=False,  timeout=0.03,  background=None,  image_base_folder="images",  bg_sound="intonation.wav",  fps=60): 
        if not fullscreen: 
            super(Scene,  self).__init__(width,  height,  fullscreen=False,  resizable=resizable,  caption=scene_name)
        else: 
            super(Scene,  self).__init__(fullscreen=fullscreen,  caption=scene_name)
        self.scene_name = str(scene_name)
        # Add a background, if one is given. 
        self.background = background #: The background of the window. 
        if self.background is not None: 
            self.background = image.load(self.media_path(image_base_folder,  background))
        self.timeout = timeout #: The timeout between runs through the event loop
        # First we enable OpenGL for alpha channel
        glEnable(GL_BLEND)
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
        
        # The we add frames per second
        self.frames_per_second = fps #: The number of frames to display per second
        self.frame_display_time = 1.0 / self.frames_per_second #: The amount of time a frame should be displayed. 
        self.delta_t = 0 #: The amount of time which passed since the last time the frame changed. 
        
        # Then we add some sound
        
        #sound = media.load(self.media_path("music",  bg_sound),  streaming=False)
        #sound.play()
        
        # now we also specify a keyboard event handler
        self.keyboard = key.KeyStateHandler() # this can be used anywhere by asking if self.keyboard[UP]... doesn't yet work as it should. 
        # The subclass has to implement the event_loop for flexibility reasons. 
    
    def media_path(self,  base_folder,  source):
        """Get the full path from base_folder and source name."""
        return path.join(path.join(path.dirname(__file__),  base_folder),  source)
    
    def on_key_press(self,  symbol,  modifiers):
        """React to Keyevents. this needs a protagonist. 
        
        First: Basic movement."""
        # First get us the basic actions from pyglet. 
        super(Scene,  self).on_key_press(symbol,  modifiers)
        # If the Arrowkey Up was pressed, try to set move up of the protagonist to one, so we get smooth movement. 
        # If no protagonist is defined, just do nothing. 
        if symbol == key.UP: 
            try: 
                self.protagonist.y_move_up = 1
            except: pass
        # On arrowkey down, tell the protagonist to move down on next move. 
        elif symbol == key.DOWN: 
            try: 
                self.protagonist.y_move_down = 1
            except: pass
        # On arrowkey right, tell the protagonist to move right on next move. 
        elif symbol == key.RIGHT: 
            try: 
                self.protagonist.x_move_right = 1
            except: pass
        # On arrowkey left, tell the protagonist to move left on next move. 
        elif symbol == key.LEFT: 
            try: 
                self.protagonist.x_move_left = 1
            except: pass
            
            
    
    def on_key_release(self,  symbol,  modifiers):
        """React to Keyevents. 
        
        First: Basic movement."""
        # If the UP key was released, tell the protagonist to move no longer. 
        # If no protagonist exists just do nothing. 
        if symbol == key.UP: 
            try: 
                self.protagonist.y_move_up = 0
            except: pass
        # Reset movement down
        elif symbol == key.DOWN: 
            try: 
                self.protagonist.y_move_down = 0
            except: pass
        # Reset movement right
        elif symbol == key.RIGHT: 
            try: 
                self.protagonist.x_move_right = 0
            except: pass
        # Reset movement left
        elif symbol == key.LEFT: 
            try: 
                self.protagonist.x_move_left = 0 
            except: pass
        
    # Now we need an action: 
    def action(self):
        """Do what the script should do in the event loop.
        
        For example you could implement a method which tells all Things to act, move and show themselves. 
        
        for i in actors: # actors is a list of all Things to show :: 
            i.act()
            i.move()
            i.blit()
        """
        raise NotImplementetException("An action in the event_loop must be implemented for a scene to do anything.")
        
    # Later on, we'll also need the event loop. 
    def event_loop(self):
        while not self.has_exit: 
            self.dispatch_events()
            #media.dispatch_events()
            
            # Increase the timer by the fractions of seconds which passed to see if a new frame should be displayed. 
            self.delta_t += clock_tick()
            # If it's time to display another frame, update the screen. 
            if self.delta_t >= self.frame_display_time: 
                # Clear the screen
                self.clear()
                # Show the background, if we have one. 
                if self.background is not None: 
                    self.background.blit(0, 0, -1)
                # Do whatever action should be done by default. 
                self.action()
                # And update the Scene with the data generated by self.action()
                self.flip()
                # At the end, set the amount of time passed since last frame to 0 again. 
                self.delta_t = 0
            # And wait 1ms. This means a maximum of 1000 loops per second to save processor load. 
            sleep(self.timeout)
        

# We'll also need things. 

class Thing(object):
    """Every Thing we want to show. 
    
    Every Thing we want to show has an image and a location, can move() and act() and detect if it touches(other) an other Thing. Also it can wobble away from any Thing it touches when called to avoid_collisions_fuzzy(other_things)."""
    def __init__(self,  x=240,  y=320,  image_source="ball-ally.png",  image_dir="images",  visible=True): 
        self.image_source = image_source #: The name of the image to show. 
        self.x = x #: The x position of the part
        self.y = y #: The y position of the part
        self.x_move_right = 0 #: If this contains a value, the Thing moves the value into this direction per round
        self.x_move_left = 0 #: If this contains a value, the Thing moves the value into this direction per round
        self.y_move_up = 0 #: If this contains a value, the Thing moves the value into this direction per round
        self.y_move_down = 0 #: If this contains a value, the Thing moves the value into this direction per round
        self.image_base_folder = path.join(path.dirname(__file__),  "images") #: The folder where we store the images. 
        self.image = image.load(path.join(self.image_base_folder,  self.image_source)) #: The image with all its functions. 
        self.width = self.image.width #: The width of the thing added for convenience. 
        self.height = self.image.height #: The height of the thing added for convenience. 
        self.visible = visible #: Set, if the image should be shown. 
        
    def act(self): 
        """Do whatever the Thing should do per round by default."""
        pass
        # raise NotImplementetException("Every Thing must have a default action - a defined act() method.")
        
    def move(self):
        """If the Thing has to move, it does. Variables: self.x_move and self.y_move."""
        self.x += self.x_move_right
        self.x -= self.x_move_left
        self.y += self.y_move_up
        self.y -= self.y_move_down
    
    def keep_on_screen(self, width, height): 
        """If the thing left the screen, put it back."""
        # if it's out the left corner, put it at the left corner
        if self.x < 0: 
            self.x = 0
        # If ifs out rigth, put it at the border with its right side. 
        elif self.x > width - self.width: 
            self.x = width - self.width
        # If its below the bottom, move it to the bottom
        if self.y < 0: 
            self.y = 0
        # if it's above the top, put it back so that it exactly touches the top. '
        elif self.y > height - self.height: 
            self.y = height - self.height
    
    def touches(self,  other):
        """Check if the Thing touches an other Thing."""
        return touches(self,  other)
    
    def avoid_collisions_fuzzy(self,  others,  mode="step_back"):
        """Move away, if you collide with one or several of the others. 
        
        Call it after each move with all other Things with which to avoid collisions to get fuzzy collision avoidance. 
        
        If the Thing touches an other Thing when this method is called, it wobbles a random range away from it. 
        
        Ideas: 
            - Add a weight parameter. A thing should only step back, if it has about the same or lower stability / weight. 
        """
        if mode == "step_back": 
            for k in others: 
                # touches it
                if self.touches(k): 
                    # and if it does, it moves one step away. 
                    if self.x == k.x: 
                        self.x += randrange(-1,  1)
                    elif self.x > k.x: 
                        self.x += randrange (0,  self.width)
                    elif self.x < k.x: 
                        self.x -= randrange (0,  self.width)
                    # same for y
                    if self.y == k.y: 
                        self.y += randrange(-1,  1)
                    elif self.y > k.y: 
                        self.y += randrange (0,  self.height)
                    elif self.y < k.y: 
                        self.y -= randrange (0,  self.height)
    
    def move_random(self): 
        """Move at random, this means, wobble a bit. 
        
        The distance of one wobble should be equal to 1/10th of width and height"""
        # To have the random movement a bit bell shaped, do it thrice. 
        for i in range(3): 
            self.x += randrange(-1,  2) * self.width / 10 # randrange (-1, 2) mostly stays at one place <x> = 0
            self.y += randrange(-1,  2) * self.height / 10 # I'm sure it has a reason... 
    
    def blit(self): 
        """Show the Thing, if it's visible. 
        
        If self.visible is true, show the Thing at its coordinates (self.x, self.y). To hide a Thing, just set self.visible to False. """
        if self.visible: 
            self.image.blit(x=self.x,  y=self.y)
        else: 
            pass

# And a useful function to determine, if some thing touches some other thing. 

def edges_touch(thing,  other):
    """Check if the thing exactly touches a corner or an edge of the other. """
    if thing.x == other.x and thing.y == other.y or thing.x == (other.x + other.width) and thing.y == other.y or thing.x == (other.x + other.width) and thing.y == (other.y + other.height) or thing.x == other.x and thing.y == (other.y + other.height) or (thing.x + thing.width) == other.x and thing.y == (other.y + other.height) or (thing.x + thing.width) == other.x and thing.y == other.y or (thing.x + thing.width) == other.x and (thing.y + thing.height) == other.y or thing.x == other.x and (thing.y + thing.height) == other.y or thing.x == (other.x + other.width) and (thing.y + thing.height) == other.y: 
        return True
    else: 
        return False

def a_corner_is_inside(thing, other):
    """Check if one of the edges of other is inside thing"""
    # Lower left corner of other == upper right corner of thing
    if thing.x <= other.x and thing.x + thing.width >= other.x and thing.y <= other.y and thing.y + thing.width >= other.y: 
        return True
    # lower right corner of other
    elif thing.x <= other.x + other.width and thing.x + thing.width >= other.x + other.width and thing.y <= other.y and thing.y + thing.height >= other.y: 
        return True
    # upper right corner of other
    elif thing.x <= other.x + other.width and thing.x + thing.width >= other.x + other.width and thing.y <= other.y + other.height and thing.y + thing.height >= other.y + other.height:
        return True
    # upper left corner of other
    elif thing.x <= other.x and thing.x + thing.width >= other.x and thing.y <= other.y + other.height and thing.y + thing.height >= other.y + other.height: 
        return True
    else: 
        return False

def an_edge_is_inside(thing,  other):
    """Check if an edge of thing is inside other."""
    return a_corner_is_inside(other,  thing)


def touches(thing,  other,  mode="squares"):
    """Test is something touches some other thing. 
    
    Check if the given coordinates (optionally: A part of the given other) is in the other. 
    
    If the given thing or some part of the thing touches the other or is inside the other.
    
    
    Ideas: 
        - Implement other modes, for example "circle"
        
        """
    # Check if the coordinates or area exactly touch a corner or an edge of the other. 
    if mode == "squares": 
        if edges_touch(thing,  other): # one of the corners or edges touches. 
            return True
        
        # Lower left of other inside thing 
        elif a_corner_is_inside(thing,  other):
            return True
        # Now we only need to check, if a part other defined by pos_x + width and pos_y + height is inside the given other. 
        # We can begin with the knowledge, that the lower left edge isn't inside the other (see previous if). 
        # Now we only need to check, if the edge is left of it or lower. If the edge + width/height isn't also left of it or lower, we have a hit. 
        # check if the exact position for x and y are both smaller and the width/height is larger than teh others values
        elif an_edge_is_inside(thing,  other): 
            return True
        # since we already know that the pos_x and pos_y are not inside the ares, we just have to consider the last case: 
        # The defined other is not inside other. 
        else: 
            return False

def remove_invisible_things(list_of_things):
    """Remove all invisible things from the list. 
    
    The list is modified, not copied. 
    
    The return type is None, just as it is for the builtin list functions."""
    # Check for each thing in the list, if it is visible. If it isn't, remove it. 
    for i in list_of_things: 
        # If it's not visible, remove it. 
        if not i.visible: 
            list_of_things.remove(i)

#### Self-Test ####

def _test(): 
    """Do Doctests."""
    from doctest import testmod
    testmod()
    
def _basic_use_case():
    """Test the basic use case: Subclass Scene with two actors. """
    # First create two blobs
    thing = Thing(x=110,  y=120)
    thang = Thing(x=100,  y=100)
    
    # Now add both to a list of blobs. 
    blobs = [] #: This list contains all blobs. 
    blobs.append(thing)
    blobs.append(thang)
    
    # Subclass Scene. 
    class Test_Scene(Scene): 
        def __init__(self): 
            # Get the Scene __init__
            super(Test_Scene,  self).__init__()
            
            # Define the actors. 
            self.actors = blobs
            # And the actor to be moved with the arrowkeys. 
            self.protagonist = blobs[0]
            
            # Start the event loop. 
            self.event_loop()
        
        # Now define the basic action
        def action(self): 
            # tell every actor
            for i in self.actors: 
                # to move
                i.move()
                
                # Then do collision detection. 
                # copy the list of actors
                others = self.actors[:]
                # And remove yourself
                others.remove(i)
                # Now avoid collisions with all others
                i.avoid_collisions_fuzzy(others)
                
                # In the end show yourself. 
                i.blit()
    
    # call the Test_Scene
    scene = Test_Scene()

# if this file is called directly, we just show two blobs as test-case. 

if __name__ == "__main__": 
    # Just test the basic use case
    _basic_use_case()
    
    # And while we're at it: do doctests
    _test()
    
    
