#/ust/bin/env python
# encoding: utf-8

"""A battle between two Blobs using the One Die roleplaying System. 

Plans: 
    - TODO: Show which blobs is controlled by the player (the protagonist)
    - TODO: Ask the player for an identifier and a name for his fighter inside pyglet. This might require creating a text_environment, first. 
    

Ideas: 
    - Also decide how you attack or defend. 
    - Add "speech bubbles" above the blobs. 
    - Two different modes: First "test your enemy" with little chance to get wounded (don't try to wound the other: +6) to find out the strategy of the other, afterwards real battle. If you switch too early, you have little chances to hit the enemy, but every step you switch too late carries the remote chance that you might be hit yourself. +6 obly means a chance of 5/6 not to be hit by a similarly abled foe. 


Important notes: 
    - *args and **kwds in __init__() makes sure, that we aren't confused, when we get more arguments than we expected. 
    
    - We now need to call super with the *args and the **kwds, so superclasses called alongside Being (i.e. char.Char) get the arguments, because the superclasses get put into a chain in which the arguments are passed along until some class catches them (only those are called *args and **kwds which aren't explicitely in __init__() of the class). 
    """


#### Imports ####

# First we import basic simplifications. 
import babglet

# Import the improvised keypad via the arrowkeys
import babpad

# Also get the dicts which show the effects of positions on the attack roll. 
import babbattle_positions

# Now we get necessary modules from pyglet
from pyglet import image

# And the keyboard controls
from pyglet.window import key

# And we want to be able to show text
from pyglet import font

# And the path module to be able to create platform independent filepaths
from os import path

# As well as the square root. 
from math import sqrt

# And some random movement. 
from random import randrange

# For automatic selection of decisions, we also need the choice module. 
from random import choice

# Additionally we import the full blown character management of the EWS RPG system. 
from ews import char


#### CONSTANTS ####

SCENE_NAME = "Single Blob Battle"

IMAGE_BACKGROUND = "bg4-1200-800.png" #: The Background image. 

BASE_IMAGE = "blob_zeichnung-2-geschnitten_blob_weiche_maske.png" #: The image of a Blob. 

# IMAGE_WEAPON = "ball-ally.png" #: The image showing the weapon. 
IMAGE_WEAPON = "blob_zeichnung-2-geschnitten_schwert_weiche_maske_klein.png" #: The image showing the weapon. 

IMAGE_WEAPON_TEMP = "ball-ally_transparent.png" #: The image showing the temporary position of the weapon (if we need to define two positions). 

IMAGE_WEAPON_PREV = "ball-ally_transparent_red.png" #: The image showing the previous position of the weapon. 

IMAGE_MARKER_HIT = "blobf.png" #: The image showing that a Being was hit. 

IMAGE_MARKER_ATTACKER = "mark_attacker.png" #: The image showing who is the attacker. 


# First we need the battleground

class Scene(babglet.Scene):
    """The Scene where we want action."""
    def __init__(self,  actors = [], *args, **kwds): 
        super(Scene,  self).__init__(background=IMAGE_BACKGROUND, scene_name=SCENE_NAME, *args, **kwds)
        # First we need a state of the program. the basic state is "welcome"
        self.state = "battle"
        # Now we need the actors
        self.actors = actors
        for i in self.actors: 
            i.host = self
        # And the figure to move: 
        self.protagonist_being = actors[0]
        self.protagonist = self.protagonist_being.base #: This is the Thing which will be moved with the keyboard. 
        self.protagonist.enemy = self.prepare_enemy(self.protagonist,  "defender") #: The enemy of the protagonist. 
        
        # Now show a bit of text
        # First load the font definition
        self.font = font.load("Arial", 12)
        # The create the text for story content. 
        self.text_story = font.Text(self.font, x=7, y=10)
        # And fill in a greeter text. 
        self.text_story.text = "Greetings, "+ self.protagonist_being.name + ", \nYou will have to prevail in this battle. \nThe first strike shall be yours." + "\n" + "Please select with the arrow keys or the number pad where you want to attack. Confirm with Enter or Return."
        # Make the text_storys width to 2/3rd of the Scene. 
        self.text_story.width = self.width * 2.0/3 - 10
        # Reposition the text_story, so that it is shown completely. 
        self.text_story.y = self.text_story.height
        
        # Now also add a battle-text with details about the battle. 
        # First we need a smaller font. 
        self.font_small = font.load("Arial", 10)
        # Now add the detail_text
        self.text_details = font.Text(self.font_small, x=self.width*2.0/3+10, y=10)
        # Give it some greeter text
        self.text_details.text = "Details about the Battle"
        # And set its width to 1/3rd of the Scene. 
        self.text_details.width = self.width/3.0 - 10
        
        # Add a game_over screen, a really large font. 
        self.font_banner = font.load("Arial", 48)
        # Now add the banner text, centered on screen
        self.text_banner = font.Text(self.font_banner, x=self.width/6.0, y=self.height/3.0, halign=font.Text.CENTER)
        # Now add the text
        self.text_banner.text = "Game Over"
        # and add a width and height: 2/3 of the screen. 
        self.text_banner.width = self.width * 2.0 / 3
        
        
    
    def on_key_press(self,  symbol,  modifiers):
        super(Scene,  self).on_key_press(symbol,  modifiers)
        # If we're still in the battle, react to keypad input. 
        if self.state == "battle": 
            babpad.keypad(symbol,  self.protagonist)
        
    def on_key_release(self,  symbol,  modifiers):
        super(Scene,  self).on_key_release(symbol,  modifiers)
    
    def show_parts(self): 
        """Show all parts of the scene."""
        for i in self.actors: 
            i.blit()
        self.text_story.draw()
        # Only show the battle text while we're in battle
        if self.state == "battle": 
            self.text_details.draw()
        # If the game is finished, show the text_banner
        if self.state == "won" or self.state == "lost": 
            self.text_banner.draw()
    
    def action(self): 
        """Do what should happen. """
        if self.state == "battle": 
            
            #self.protagonist.x = 300
            #self.protagonist.y = 100
            # First act and move
            for i in self.actors: 
                i.move()
                i.act(self.actors)
            # Then show. 
            self.show_parts()
            
            # Check if both are still alive TODO: Make the code nicer, i.e. just let both fighters evoke the die() method. 
            # First the protagonist
            self.protagonist.being.checkalive()
            if not self.protagonist.being.active: 
                # If the protagonist is dead, hide it. 
                self.protagonist.visible = False
                # And say so. 
                self.update_text(story_append="\nYou had to cease fighting.", banner_append="\nYou lost.")
                self.state = "lost"
            
            # Then the enemy
            self.protagonist.enemy.checkalive()
            if not self.protagonist.enemy.active: 
                # If the enemy is dead, hide it. 
                self.protagonist.enemy.base.visible = False
                # And say so. 
                self.update_text(story_append="\nYour enemy had to cease fighting.", banner_append="\nYou won.")
                self.state = "won"
                
        if self.state == "won" or self.state == "lost": 
            self.show_parts()
           
           
    def prepare_enemy(self,  player,  role):
        """Let the player fight against the last of the actors"""
        self.actors[-1].fight_role = role
        return self.actors[-1]
    
    
    def update_text(self, story=None, story_append=None, details=None, details_append=None, banner=None, banner_append=None):
        """Update the content of the text_story."""
        # First check if things should be appended. 
        # If yes, change the content accordingly. 
        # First for the story text
        if story_append is not None: 
            # append the append to the main text. 
            # If we got no story, we justa ppend to the existing text. 
            if story is None: 
                story = self.text_story.text + story_append
            # Else we append the append to the story. 
            else: story += story_append
        # the same for details 
        if details_append is not None: 
             if details is None: 
                 details = self.text_details.text + details_append
             else: details += details_append
        # And for the banner
        if banner_append is not None: 
             if banner is None: 
                 banner = self.text_banner.text + banner_append
             else: banner += banner_append
        
        # Check if we got a story text part
        if story is not None: 
            self.text_story.text = story
            # Reposition the text_story, so that it is shown completely. 
            self.text_story.y = self.text_story.height
        # Also check if we got a details text part
        if details is not None: 
            self.text_details.text = details
            # Reposition the text_details, so that it is shown completely. 
            self.text_details.y = self.text_details.height
        # And check for banner text
        if banner is not None: 
           self.text_banner.text = banner
           # Reposition the text_details, so that it is shown completely. 
           self.text_banner.y = self.text_banner.height
            
            
    def save(self): 
        """Save the game data. As first step it consists of all actors, which now get saved."""
        for i in self.actors: 
            i.save()


# Then we need a fighter. 
# We create a Being from which we subclass Fighter, because that is the most general type of thing which fights. 

class Being(object):
    """A being of some kind. It has body parts and might be able to move. 
    
    *args and **kwds makes sure, that we aren't confused, when we get more arguments than we expected. 
    
    We now need to call super with the *args and the **kwds, so superclasses called alongside Being (i.e. char.Char) get the arguments, because the superclasses get put into a chain in which the arguments are passed along until some class catches them (only those are called *args and **kwds which aren't explicitely in __init__() of the class). 
    """
    def __init__(self,  x=640,  y=480,  host=None,  marker=None, *args,  **kwds):
        super(Being,  self).__init__(*args,  **kwds)
        self.host = host #: The Scene in which the Being lives. 
        self.base = Part(x=x,  y=y, being=self) #: This is the main part of the body. 
        # Now the distance of the limbs should be the square root of the squared width and height of the base, so that we can support arbitrary bases. 
        self.distance_of_limbs = sqrt(self.base.width**2 + self.base.height**2) #: Distance of limbs in x and in y direction
        
        self.parts_to_show = [self.base] #: All parts of the Being. Parts decide themselves, if they want to be shown (thing.visible)
        
        self.blah = 0 #: Multi purpose development variable used for about anything and MUST be replaced at the end. TODO: replace any instance of this before making a real release. 

    
    def move(self):
        """Move all parts."""
        # Just tell them to do their movement
        for i in self.parts_to_show: 
            i.move()
        # Position the base close to the base of the enemy, height below. 
        # TODO: Change to a relative pos, so the blob moves gradually. 
        if self.base.enemy is not None: 
            # Same x as the enemy - face him/her
            # the ideal x would be
            target_x = self.base.enemy.base.x
            # the ideal y would be
            # 40px distance from its lower border. 
            target_y = self.base.enemy.base.y - self.base.height  - 40
            # Now half the distance to that ideal position. 
            # That means, move into a distance equal to 
            # half the distance between x pos and target x pos 
            # vektor: + (target - current) / 2
            self.base.x += (target_x - self.base.x) / 2
            self.base.y += (target_y - self.base.y) / 2
            
        # And add some random movement, so we're not too static :) 
        self.base.move_random()
        
        # At the end make sure that they keep on the screen (Scene)
        for i in self.parts_to_show: 
            i.keep_on_screen(self.host.width, self.host.height)
        
        
    def blit(self):
        """Show all parts."""
        for i in self.parts_to_show: 
            i.blit()
            
    
    def act(self,  actors): 
        """Do what you normally do when asked to do something."""
        # First copy the list of actors
        others = actors[:]
        
        # Then remove yourself. 
        others.remove(self)
        
        # If a body part touches the body part of some other, move it away from it. 
        self.avoid_all_collisions(others)
        
        # Move the body parts a bit back to where they belong. 
        self.reposition_limbs() 
        
    
    def reposition(self,  thing,  position):
        """Put the thing into a new position. 
        
        Positions can be called with the number-pad. 
        
        Positions are: top (8), upper right (9), upper left (7), left (4), right (6), lower left (1), lower right (3). 
        
        """
        # Change the x value of the thing relative to its base. 
        thing.x_rel = self.distance_of_limbs*self.positions_list[position][0]
        # Change the y value of the thing relative to its base. 
        thing.y_rel = self.distance_of_limbs*self.positions_list[position][1]
        
        # Set the things position info to the new position. 
        thing.position = position


    def reposition_limbs(self,  mode="swiftly",  count_invisible=True):
        """Move the limbs where they belong.
        
        count_invisible sets, if invisible parts should be taken into account for the weighing between base and parts. """
        # Now check, if the body parts are the right distance from the base. If not, move both a bit. 
        
        # First copy the list of parts of the Being
        parts_without_base = self.parts_to_show[:]
        parts_to_consider = self.parts_to_show[:]
        # Then remove the base part relative to which the other parts set their positions. 
        parts_without_base.remove(self.base)
        
        # If we don't want to consider invisible parts, we need to remove all invisible parts
        # From the list of parts top consider. 
        # Check all parts and remove the ones which are not visible. 
        if not count_invisible: 
            # Remove all invisible things from the list with the function in babglet
            babglet.remove_invisible_things(parts_to_consider)
            babglet.remove_invisible_things(parts_without_base)
        
        # Use the correct mode to reposition the limbs
        if mode == "swiftly": 
            self.reposition_limbs_swiftly(parts_to_consider,  parts_without_base)
        elif mode == "wobbly": 
            self.reposition_limbs_wobbly(parts_to_consider,  parts_without_base)
            
    
    def reposition_limbs_swiftly(self,  parts_to_show,  parts_without_base):
        """Move the limbs swiftly where they belong.
        
        The parts move a bit to their taget position. The base also moves a bit into the opposite direction, so they "meet in the middle" (with their relative position now exact again). 
        """
        # Check if all limbs are in the correct position. 
        # If they aren't, move them a part towards the correct position 
        # And move the base the remaining (smaller) part, so that the part is in the correct position. 
        for i in parts_without_base: 
            if i.x - self.base.x == i.x_rel: 
                pass
            # If the limb is too far away or too close
            else:  
                # Move the limb to the correct position. 
                i.x -= float((i.x - self.base.x) - i.x_rel)  * len(parts_without_base) / len(parts_to_show)
                # Also move the base, but range divided by number of parts (for weight). 
                self.base.x += float((i.x - self.base.x) - i.x_rel)  / len(parts_to_show)
            if i.y - self.base.y == i.y_rel: 
                pass
            # If the limb is too far away or too close
            else:  
                # Move the limb to the correct position. 
                i.y -=  float((i.y - self.base.y) - i.y_rel) * len(parts_without_base) / len(parts_to_show)
                # Also move the base, but range divided by number of parts (for weight). 
                self.base.y +=  float((i.y - self.base.y) - i.y_rel)  / len(parts_to_show)
                

    def reposition_limbs_wobbly(self, parts_to_show, parts_without_base):
        """Move the limbs wobbly where they belong.
        
        This could be used for drunken chars."""
        # Check if all limbs are in the correct position. 
        # If they aren't, move them a part towards the correct position 
        # And move the base the remaining (smaller) part, so that the part is in the correct position. 
        for i in parts_without_base: 
            if i.x - self.base.x == i.x_rel: 
                pass
            # If the limb is too far away or too close
            else:  
                # Check the distance from the target position
                distance_to_position = float((i.x - self.base.x) - i.x_rel) 
                # calculate a random step towards that direction. 
                random_step = self.random_step_with_max_range(distance_to_position)
                # Move the limb a bit towards the correct position. 
                i.x -= random_step * len(parts_without_base) / len(parts_to_show)
                # Also move the base, but range divided by number of parts (for weight). 
                self.base.x += random_step / len(parts_to_show)
            if i.y - self.base.y == i.y_rel: 
                pass
            # If the limb is too far away or too close
            else:  
                # Check the distance from the target position
                distance_to_position = float((i.y - self.base.y) - i.y_rel) 
                # calculate a random step towards that direction. 
                random_step = self.random_step_with_max_range(distance_to_position)
                # Move the limb a bit towards the correct position. 
                i.y -=  random_step * len(parts_without_base) / len(parts_to_show)
                # Also move the base, but range divided by number of parts (for weight). 
                self.base.y +=  random_step / len(parts_to_show)
                

    def random_step_with_max_range(self,  distance):
        """Calculate a random step in the given direction with a maximum equal to the full distance."""
        if distance > 1: 
            random_step = randrange(0,  int(distance))
        elif distance > 0: 
            random_step = randrange(0,  1)
        elif distance < -1: 
            random_step = - randrange(0, - int(distance))
        elif distance < 0: 
            random_step = - randrange(0, 1)
        else: 
            random_step = 0
        return random_step
        

    def avoid_all_collisions(self, others,  avoid_invisible=True):
        """This moves each part of the Being out of the way of other parts."""
        # each body part now
        for i in self.parts_to_show: 
            # checks for each of the other beings
            for j in others: 
                # if one of its parts collides, and try to "avoid" the collision. 
                # If we should avoid invisible parts, just avoid collisions with all parts
                if avoid_invisible: 
                    i.avoid_collisions_fuzzy(others=j.parts_to_show)
                # If it shouldn't avoid invisible parts, first check for each part, if it is visible. 
                else: 
                    other_parts = j.parts_to_show[:]
                    babglet.remove_invisible_things(other_parts)
                    i.avoid_collisions_fuzzy(others=other_parts)




# for that Being we want parts. A part is just a thing, so we get a thing. . 

class Part(babglet.Thing):
    """A part of a Being, Different from a Thing, a part has a base and a relative x position, a relative y position and a position identifier, and it knows to which Being it belongs. 
    
    Every Part has an image and a location as defined in the thing."""
    def __init__(self,  base=None,  x=240,  y=320,  image_source=BASE_IMAGE,  x_rel=0,  y_rel=0, being=None): 
        self.being = being #: The Being to which teh part belongs. 
        self.base = base #: The base is the center of the being to which the part belongs. 
        self.x_rel = x_rel #: x relative to the base
        self.y_rel = y_rel#: y relative to the base
        self.position = None #: A position identifier
        
        # If we get no external base, we are the base. 
        if base is None: 
            # We just create a Thing without special mods for being a part. 
            super(Part,  self).__init__(x=x,  y=y,  image_source=image_source)
            self.base = self 
        # Else we create a Thing with base and relative positions. 
        else: 
            super(Part,  self).__init__(x=self.base.x  + self.x_rel,  y=self.base.y  + self.y_rel,  image_source=image_source)
        


# And now we subclass the Being to a Fighter. 

class Fighter(Being,  char.Char): 
    """A Fighter. 
    
    A Fighter has a weapon and can attack, decide on a weapon position and the weapon position can be controlled with the arrow keys and the number keypad.
    
    @param fight_role: The role the char assumes in the battle. 
    @type fight_role: A string describing teh role. At the moment it takes either "attacker" or "defender". 
    
    """
    def __init__(self,  x=640,  y=480, fight_role="attacker",  host=None,  enemy=None,  marker=None,  weapon_initial_position=9, source="tag:1w6.org,2008:Menschen", template=True): 
        # First get the init method of the Being and of the char.Char. 
        # In this we pass on the arguments of the fighter function which are useful to a being. 
        super(Fighter,  self).__init__(x=x,  y=y,  host=host,  marker=marker, source=source,  template=template)
        
        self.base.reposition_weapon = self.reposition # Just pass on the whole function, so it can be used by the protagonist. 
        self.base.choice_buffer = self.choice_buffer # Just pass on the whole function
        self.choice_buf = None #: A buffered part of a choice if a string, or the buffered choice, if an int. 
        
        #: weapon positions are lists with x, y, both relative to the main char
        self.positions_list = {1: [-1,  -1],  2: [0,  -1],  3: [1,  -1],  4: [-1,  0],  5: [0,  0],  6: [1,  0],  7: [-1,  1],  8: [0,  1],  9: [1,  1]}
        
        # Now define the limbs with their relative positions to the base (x_rel and y_rel). 
        self.weapon = Part(base = self.base,  x_rel = self.distance_of_limbs,  y_rel = self.distance_of_limbs,  image_source=IMAGE_WEAPON) #: This is the weapon. Pass it the image_source parameter with a name of a source image to make it look differently. 
        
        # And move the weapon to the initial position. 
        self.choice_buffer(decision=weapon_initial_position)
        
        # Now define a temporary marker for the first choice
        #: A temporary marker to show which position was selected. 
        self.weapon_temp = Part(base = self.base,  x_rel = self.distance_of_limbs,  y_rel = self.distance_of_limbs,  image_source=IMAGE_WEAPON_TEMP)
        # Hide the temporary marker at the moment. 
        self.weapon_temp.visible = False
        
        # And define a marker for the previous choice. 
        #: The previous position of the weapon. 
        self.weapon_prev = Part(base = self.base,  x_rel = self.distance_of_limbs,  y_rel = self.distance_of_limbs,  image_source=IMAGE_WEAPON_PREV)
        # At the beginning, set its position to the current position
        self.weapon_prev.position = self.weapon.position
        self.fight_pos_prev = self.weapon.position
        
        self.fight_role = fight_role #: The role it assumes. Normally either attacker or defender. 
        self.fight_choice_1 = None #: The choice of position_1. An Attacker has only this choice, 
        self.fight_choice_2 = None #: The choice of position 2. For a defender this signifies the place where you strike a counterattack, if you manage to defend. 
        
        # Add the weapon, the temporary position and the previous position of the weapon to the parts. 
        self.parts_to_show.append(self.weapon)
        self.parts_to_show.append(self.weapon_temp)
        self.parts_to_show.append(self.weapon_prev) 
        
        # Add a marker, if the Being is attacker or defender
        self.marker_role = Part(base = self.base,  x_rel = - self.distance_of_limbs,  y_rel = self.distance_of_limbs,  image_source=IMAGE_MARKER_ATTACKER) #: A marker which shows, if the Being is an attacker of a defender. 
        
        # Set the correct marker (attacker or defender). This will be set again each turn. 
        self.set_fight_role_marker(mark=self.fight_role)
        
        # Also add 
        #: A marker which shows if the Being was hit
        self.marker_hit = Part(base = self.base,  x_rel = self.distance_of_limbs * 2,  y_rel = 0,  image_source=IMAGE_MARKER_HIT)
        
        # And hide the hit marker
        self.marker_hit.visible = False
        
        #: the time specifying how long to show the hit marker. 
        self.marker_hit_display_time = 0 
        
        # Also add the role marker and the hit marker to the list of parts. 
        self.parts_to_show.append(self.marker_role)
        self.parts_to_show.append(self.marker_hit)
        

        # Set the fight role and show the marker. 
        self.set_fight_role(fight_role) 

        # Now add the enemy. 
        self.base.enemy = enemy
        
        # And add a battle state, which shows if we are just "test"ing the strength of our enemy, or if we want to really do damage. 
        # self.state_of_the_battle = "testing the strength"
        self.state_of_the_battle = "battle for life"
        
        
    def act(self,  actors):
        """Do what you do by default when called to just act."""
        # First get the act method of the superclass Being. 
        super(Fighter,  self).act(actors)
        
        # then add Fighter-specific actions
        
        # Set the correct marker (attacker or defender)
        self.set_fight_role_marker(mark=self.fight_role)
        
        # Unshow the hit-marker, so that it only flickers once. 
        # TODO: Set a time to show using clock.step() or so. 
        if self.marker_hit.visible and self.marker_hit_display_time > 10: 
            self.marker_hit.visible = False
            self.marker_hit_display_time = 0
        elif self.marker_hit.visible: 
            self.marker_hit_display_time += 1
        
        
    def set_fight_role(self,  role="attacker"):
        """Set the fight role of the being."""
        self.fight_role = role
        self.set_fight_role_marker(self.fight_role)
    
    def set_fight_role_marker(self,  mark=None):
        """Set the marker which shows, if the Being is attacker or defender."""
        # print "Set fight_role",  self.name,  mark
        
        # If the role maker is not yet set to the corrrect image, set it. 
        if self.marker_role is None: 
            self.marker_role = Part(base = self.base,  x_rel = - self.distance_of_limbs,  y_rel = self.distance_of_limbs,  image_source=IMAGE_MARKER_ATTACKER)
        
        # If we're defender, don't show the marker. 
        if mark == "defender" and self.marker_role.visible: 
            # Remove the role maker from the list of parts to show. 
            self.marker_role.visible = False
            
        # If we're an attacker, show the role maker. 
        if mark == "attacker" and not self.marker_role.visible: 
            
            # Add the role marker to  the list of parts to show. 
            self.marker_role.visible = True
    
    
    def auto_decide(self, strategy="clockwise"):
        """Decide automatically which strategy to take"""
        # Without tactic, we just choose a position at random. 
        if strategy is None: 
            # If we're an attacker, we need to take one decision: the attack position. 
            if self.fight_role == "attacker": 
                self.fight_choice_1 = choice(self.positions_list.keys())
                self.fight_choice_2 = None
                # Move the weapon
                self.reposition(self.weapon,  self.fight_choice_1)
                # And set the "previous position" to the pos where the weapon will be when we win.
                self.reposition(self.weapon_prev,  self.fight_choice_1)
            # If we're a defender, we need to set the defend position and a counter position. 
            elif self.fight_role =="defender": 
                self.fight_choice_1 = choice(self.positions_list.keys())
                self.fight_choice_2 = choice(self.positions_list.keys())
                # Move the weapon
                self.reposition(self.weapon,  self.fight_choice_1)
                # And set the "previous position" to the pos where the weapon will be when we win.
                self.reposition(self.weapon_prev,  self.fight_choice_2)
            else: 
                raise Exception("Unknown fight role selected" + self.fight_role)
        elif strategy == "clockwise": 
            clockwise = [1, 4, 7, 8, 9, 6, 3, 2, 5]
            if self.fight_role == "attacker": 
                self.fight_choice_1 = clockwise[(clockwise.index(self.weapon.position) +1) % 9]
                self.fight_choice_2 = None
                # Move the weapon
                self.reposition(self.weapon,  self.fight_choice_1)
                # And set the "previous position" to the pos where the weapon will be when we win.
                self.reposition(self.weapon_prev,  self.fight_choice_1)
            elif self.fight_role == "defender": 
                self.fight_choice_1 = clockwise[(clockwise.index(self.weapon.position) +1) % 9]
                self.fight_choice_2 = clockwise[(clockwise.index(self.weapon.position) +1) % 9]
                # Move the weapon
                self.reposition(self.weapon,  self.fight_choice_1)
                # And set the "previous position" to the pos where the weapon will be when we win.
                self.reposition(self.weapon_prev,  self.fight_choice_2)
            else: 
                raise Exception("Unknown fight role selected" + self.fight_role)

    
    def choice_buffer(self,  choice=None,  decision=None,  save=False):
        """Decide on an action. 
        
        Buffer the choice of attack direction, so alternate choice methods can be used. 
        
        choice means: A key is pressed, which doesn't suffice to decide, but which goes one step to deciding. 
        decision means: The key sets the a possible final state with one command. 
        save means: Do we stick with the current decision? 
        """
        # Wenn eine Entscheidung gefällt wurde
        if decision is not None: 
            # setzen wir den Buffer auf die Entscheidung. 
            self.choice_buf = decision
            self.reposition(self.weapon,  decision)
        
        # If only a choice of direction was given (arrow pads), we go a two step process. 
        elif choice is not None: 
            self.choose_in_two_steps(choice)
            
        # If we want to save (Enter or Return was pressed)
        
        # And we're an attacker, and a decision was selected, we set the previous position and fight
        if save and self.choice_buf is not None and self.fight_role == "attacker": 
            # set the fight choice to the buffer
            self.fight_choice_1 = self.choice_buf
            
            # Set the decision as previous position for the next round
            self.reposition(self.weapon_prev,  self.fight_choice_1)
            # And show it. 
            if not self.weapon_prev.visible: 
                self.weapon_prev.visible = True
            # and fight. 
            self.battle_round()
            
        # For defenders, who haven't yet filled their first position and want to save
        elif save and self.fight_choice_1 is None and self.choice_buf is not None and self.fight_role == "defender": 
            # Set the first position
            self.fight_choice_1 = self.choice_buf 
            
            # Position the first selection. 
            self.reposition(self.weapon_temp,  self.fight_choice_1)
            # And show it. 
            if not self.weapon_temp.visible: 
                self.weapon_temp.visible = True
            battle_details = "First position set as " +  str(self.fight_choice_1) +  " - Counter position still needs to be set."
            self.host.update_text(details=battle_details)
            
        # For defenders with first choice, we select the second choice and fight. 
        elif save and self.fight_choice_1 is not None and self.choice_buf is not None and self.fight_role is "defender": 
            # Set the second position (the counter)
            self.fight_choice_2 = self.choice_buf 
            
            # And hide the temporary position marker, since we have a final decision (and we don't want it to linger around). 
            if self.weapon_temp.visible: 
                self.weapon_temp.visible = False
                
            # Set the decision as previous position for the next round
            self.reposition(self.weapon_prev,  self.fight_choice_2)
            # And show the previous decision marker
            if not self.weapon_prev.visible: 
                self.weapon_prev.visible = True
            
            # And fight. 
            self.battle_round()
    
    def battle_round(self):
        """One round of battle, including choosing of enemies."""
        if self.fight_role is "attacker":
            # Make the other defender and right against it. 
            enemy = self.host.prepare_enemy(self,  "defender")
        else: 
            enemy = self.host.prepare_enemy(self,  "attacker")
        enemy.auto_decide()
        self.fight_round(enemy)        
        
        # Reposition the marker of our fighter for the previous position (the NPC doesn't need it). 
        self.reposition(self.weapon_prev,  self.fight_pos_prev)
        self.fight_choice_1 = None
        self.fight_choice_2 = None
        
    def fight_round(self,  other):
        """One battle action against one char.
        
        With the result, we reposition the marker which shows the previous position of the weapon."""
        # TODO: Implement real fight instead of this stub which just changes the role.
        
        mods_self = [] #: The modifiers for the protagonists attack roll. 
        mods_other=[] #: The modifiers for the enemies attack roll. 
        
        #: The damage we get
        self.damage_self = 0
        #: Teh damage the other gets
        self.damage_other = 0
        
        # Do the rolls for both fighters. 
        attack_self = self.attack_roll(mods=mods_self)
        attack_other = other.attack_roll(mods=mods_other)
        
        # Both fighters get a modifier from their previous position to their new first position choice: 
        pos_mod_self = babbattle_positions.POSITION_CHANGE_EFFECT[self.fight_pos_prev][self.fight_choice_1] 
        attack_self += pos_mod_self 
        pos_mod_other = babbattle_positions.POSITION_CHANGE_EFFECT[self.fight_pos_prev][other.fight_choice_1] 
        attack_other += pos_mod_other 
        
        # Also, if one of them is still testing the strength of the other, he'll get 6 points bonus, but won't do any damage, even if he wins. 
        if self.state_of_the_battle == "testing the strength": 
            attack_self += 6
        if other.state_of_the_battle == "testing the strength": 
            attack_other += 6
        
        # Now one gets a bonus from the prediction of the attack position: 
        # Add the mod * 6, so it has more effect. +- 6 means, that the correct prediction leads to a 86% change to win. # TODO: Remove the * 6 multiplier for the prediction mod as soon as all mods are implemented.  
        
        # If the other is the defender
        if other.fight_role == "defender": 
            # the modifier is calculated from my attack choice and the defend choice of the other
            attack_prediction_mod = babbattle_positions.POSITION_CHANGE_EFFECT[self.fight_choice_1][other.fight_choice_1] * 6
            # the other gets a modifier from the efficiency with which it could predict the attack position (and set it as defend position). 
            attack_other += attack_prediction_mod
        # if the Fighter is the defender, it gets that mod itself. 
        elif self.fight_role == "defender": 
            # the mod is calculated from the others attack choice and my defend choice. 
            attack_prediction_mod = babbattle_positions.POSITION_CHANGE_EFFECT[other.fight_choice_1][self.fight_choice_1] * 6
            attack_self += attack_prediction_mod
        
        # Now we show a result. TODO: Put a nicer version of this into the GUI. 
        battle_details = "Own position change mod: " + str(pos_mod_self) + " [ " + str(self.fight_pos_prev) + ", " + str(self.fight_choice_1) + " ]" + ", the others pos change mod: " + str(pos_mod_other)
        
        battle_details += "\n" + "self position_1: " + str(self.fight_choice_1) +  ", other position_1: "  + str(other.fight_choice_1) + ", attack prediction modifier for defender: " + str(attack_prediction_mod)
        
        battle_details += "\n" + self.name + ", " + self.fight_role + " " + str(self.fight_choice_1) + "-" + str(self.fight_choice_2) + " - attack result: " +  str(attack_self) + ", damage: " + str(self.dam) + ", armor: " + str(self.armor)
        battle_details += "\n" + "fights against"
        
        battle_details += "\n" + other.name + " " + other.fight_role + " " + str(other.fight_choice_1) + "-" + str(other.fight_choice_2) + " - attack result: " + str(attack_other) + ", damage: " + str(other.dam) + ", armor: " + str(other.armor)
        # the print statement for the winner is set below, where also the weapons get repositioned. 
        
        # Set the previous position of the weapon for both. TODO: This should be a result of the battle. 
        # The final position of the winner should be the final position for both. 
        # If I throw higher
        if attack_self > attack_other: 
            # set my position as final position
            final_position = self.weapon_prev.position
            
            # TODO: If we're defender, Show the counter position flicker for a moment, before switching to the final position. 
            
            # and say so
            self.story_text = "You, " + self.name + ", scored a hit."
            # also show it. 
            other.marker_hit.visible = True
            
            # Now check for damage (we hit)
            # if we're just testing the strength of the other, we don't do damage
            if self.state_of_the_battle != "testing the strength": 
                # Else the damage consists of several factors. 
                # First the difference between the attack rolls. 
                self.damage_other += attack_self - attack_other
                # Then the damage of our weapon. 
                self.damage_other += self.dam
                # And substracted the armor of the other. 
                self.damage_other -= other.armor
        
            # Clean out negative damage. If the damage is below zero, the armor caught all damage. 
            if self.damage_other < 0: 
                self.damage_other = 0
            
            # Now actually do the damage. This returns a tuple: (new deep wounds, new critical wounds)
            deep_wounds_other, critical_wounds_other = other.damage(tp=self.damage_other)
        
            # Add story output with the wounds. 
            if deep_wounds_other is not 0: 
                self.story_text += "\n" + "Your enemy took a deep wound."
            elif critical_wounds_other is not 0: 
                self.story_text += "\n" + "Your enemy took a critical wound."
            
            # Also, if we were defender, we will now be attacker (attack series implementation)
            if self.fight_role == "defender": 
                # Switch the fight role
                self.switch_fight_roles()
                
            
        # If the other rolls better
        elif attack_self < attack_other: 
            # set his pos as final pos
            final_position = other.weapon.position
            
            # and say so
            self.story_text = "The enemy, " + other.name + ", scored a hit."
            # And show it. 
            self.marker_hit.visible = True
            
            # Now check for damage (the other hit)
            # if we're just testing the strength of the other, we don't do damage
            if other.state_of_the_battle != "testing the strength": 
                # Else the damage consists of several factors. 
                # First the difference between the attack rolls. 
                self.damage_self += attack_other - attack_self
                # Then the damage of our weapon. 
                self.damage_self += other.dam
                # And substracted the armor of the other. 
                self.damage_self -= self.armor
            
            # Clean out negative damage. If the damage is below zero, the armor caught all damage. 
            if self.damage_self < 0: 
                self.damage_self = 0
            
            # Now actually take the damage. This returns a tuple: (new deep wounds, new critical wounds)
            deep_wounds_self, critical_wounds_self = self.damage(tp=self.damage_self)
            
            # Add story output with the wounds. 
            if deep_wounds_self is not 0: 
                self.story_text += "\n" + "You took a deep wound."
            elif critical_wounds_self is not 0: 
                self.story_text += "\n" + "You took a critical wound. You should stop this battle!"
            
            #  If you were attacker, your attack series ends here. 
            if self.fight_role == "attacker": 
                # Switch the fight role
                self.switch_fight_roles()
                
        # If we get the same result, the attacker wins. 
        elif self.fight_role == "attacker": 
            final_position = self.weapon_prev.position
            self.story_text = "You, " + self.name
            other.marker_hit.visible = True
            
            # Now check for damage (we hit)
            # if we're just testing the strength of the other, we don't do damage
            if self.state_of_the_battle != "testing the strength": 
                # Else the damage consists of several factors. 
                # First the difference between the attack rolls. 
                self.damage_other += attack_self - attack_other
                # Then the damage of our weapon. 
                self.damage_other += self.dam
                # And substracted the armor of the other. 
                self.damage_other -= other.armor
            
        else: 
            self.story_text = "The enemy, " + other.name + ", scored a hit."
            final_position = other.weapon.position
            self.marker_hit.visible = True
            
            # Now check for damage (the other hit)
            # if we're just testing the strength of the other, we don't do damage
            if self.state_of_the_battle != "testing the strength": 
                # Else the damage consists of several factors. 
                # First the difference between the attack rolls. 
                self.damage_self += attack_other - attack_self
                # Then the damage of our weapon. 
                self.damage_self += other.dam
                # And substracted the armor of the other. 
                self.damage_self -= self.armor
        
        # Add output of the damage the respective fighter takes. TODO: Add wounds. 
        battle_details += "\nYou, " + self.name + ", take " + str(self.damage_self) + " points of damage."
        battle_details += "\n" + "Your enemy, " + other.name + " takes " + str(self.damage_other) + " points of damage."
        
        
        # Output the battle_details
        self.host.update_text(details=battle_details, story=self.story_text)
        
        # Set the prev position and the weapon position to the final position
        self.fight_pos_prev = final_position
        other.fight_pos_prev = final_position
        self.fight_pos = final_position
        other.fight_pos = final_position
        
        # Reposition the weapon and set the previous weapon position
        # other.reposition(other.weapon_prev,  other.fight_pos_prev)
        other.reposition(other.weapon,  other.fight_pos)
        self.reposition(self.weapon_prev,  self.fight_pos_prev)
        self.reposition(self.weapon,  self.fight_pos)
        
        # Also adapt the choice buffer, so the arrowkey movement fits the show position. 
        self.choice_buf = self.fight_pos
        other.choice_buf = other.fight_pos


    def switch_fight_roles(self):
        """Switch the fight roles with your enemy.
        
        The attacker becoems defender and the defender becomes attacker. 
        """
        
        # Attackers become defenders. 
        if self.fight_role == "attacker": 
            # Make ourselves a defender
            self.set_fight_role(role="defender")
            # And our enemy an attacker
            self.base.enemy.set_fight_role(role="attacker")
            # Then add story output. 
            self.story_text += "\n" + "Now it's your turn to defend yourself. Decide where you want to defend and counterattack."
        else: 
            # Make the protagonist the attacker
            self.set_fight_role(role="attacker")
            # And te enemy the defender. 
            self.base.enemy.set_fight_role(role="defender")
            # Then add story output. 
            self.story_text += "\n" + "Now it's your turn to attack. Decide where you want to attack."

    
    def choose_in_two_steps(self,  choice):
        """Alternate keypad input - just a wrapper."""
        babpad.choose_in_two_steps(choice,  self)



#### Self-Test ####

# If the script is called directly
if __name__ == "__main__": 
    # First get an identifier for the player
    player = raw_input("Please type an email address or domain name as identifier for you as player: ")
    name = raw_input("Please type a name for your char: ")
    # Create a being: the "Player"
    you = Fighter(source="tag:" + player + ",2008:" + name, template=False)
    # And another one, just for fun
    enemy = Fighter(source="tag:" + player + ",2008:Enemy", template=True)
    # And create a Scene
    battleground = Scene(actors = [you,  enemy])
    # Then start the event_loop
    battleground.event_loop()
    # When we leave, save all data. 
    battleground.save()
