#!/bin/env python
# -*- coding: utf-8 -*-

# Schlachtfeld - Großk�mpfe im EWS System 
#   http://rpg-tools-1d6.sf.net
# Copyright © 2007 - 2007 Achim Zien

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
# MA 02110-1301 USA


'''
Characters and methods for use in schlachtfeld module.
Characters act on their own now and can be imported and saved via amov module.
'''


#### IMPORTS ####

# for the fighting
from random import randrange as rnd
from random import shuffle
import math

# For selection of armies through command line arguments: 
import sys

# EWS standard dice
from ews import pmw6

# Name-Generator
import namen

# Obtain Charakters from yaml-files
from amov import Charakter, Versionsverwaltung
# replaced the alias with the real directory. 

#### IMPORTS ####




#### Top-Level Objects ####

name_objekt = namen.Name()

# COMBAT STYLES
# style configurations: (skill mod, (win own damage), (loose own damage), special pars
# damage code: (base damage, result stepsize); stepsize 10000 resembles None (statistically at least)
default_style = (0,(3,3),(3,3),None)                # default combat style: no bonusses
defensive_style = (3,(0,10000),(0,3),None)   # defensive combat style: bonus for char, but no wounds if winning and less wounds when losing.
offensive_style = (6,(3,3),(0,10000),'dead')            # offensive style: bonus for char, but death in case of failure
escape_style = (6,(3,3),(3,3),'escape')             # escape style: bonus for char, no damage to opponent, char could escape
styledict = {'defensive':defensive_style, 'offensive':offensive_style, 'default':default_style, 'escape':escape_style}

# scatter for random attack values
atscatter = [3,2,1]
#### Top-Level Objects ####




#### WRAPPERS ####

def ews():
    """Do one roll with the plusminus d6 die (wrapper for a function in pmw6)."""
    return pmw6.pmw6()

def check(skill,MW):
    """Check if a given skill test reached a min-value (wrapper for a function in pmw6)."""
    return pmw6.check(skill,MW)

def ocheck(skill,MW):
    """Return the degree of success / failure for a skill check (wrapper for a function in pmw6)."""
    return pmw6.ocheck(skill,MW)

def name_it(language="Esperanto"):
    '''Return a name corresponding to the characters language.'''
    return name_objekt.erzeuge(art=language) # ToDo: Fluszh the list of pregenerated names, when the language type changes. 
    
#### WRAPPERS ####




#### CLASSES ####

class Char:
    """A single soldier. Will be imported from either a prefab character file from amov or via a template."""
    def __init__(self,source="tag:1w6.org,2007:Mensch",template=True):
        '''Basic creation of a single soldier.'''
        # creation from amov template/char
        self.source = source
        self.ID = Versionsverwaltung.Versionen(self.source, u"Vorlagen").neuste
        self.amov = Charakter.Charakter(self.ID)
        
        # rank
        self.name_me(template)
        self.chartype = 'Soldat'
        self.description = self.amov.beschreibung
        self.sprache = self.amov.sprache
        
        # amov combat values
        #: The damage a character can take before going down. 
        self.bTP = self.amov.kampfwerte['Trefferpunkte']
        #: The wound value determines, how much damage suffices to inflict a wound on the character.  It's a List containing the value for deep wounds and for critical wounds. A deep wounds weakens the character (3 points off any skill and attribute), a critical wounds disables (the character has to roll, if he keeps standing). 
        self.WS = [self.amov.kampfwerte['Wundschwelle'],self.amov.kampfwerte['Wundschwelle'] * 3]
        #: The damage which te characters main weapon inflicts. 
        self.dam = self.amov.kampfwerte['Hauptwaffe']['Waffe']['Schaden']
        #: The attack value of the character. 
        self.attack = self.amov.kampfwerte['Hauptwaffe']['Kampffertigkeit']['Zahlenwert']
        #self.skill = self.amov.kampfwerte['Hauptwaffe']['Kampffertigkeit']['Name']
        #: The armor value of the character. 
        self.armor = self.amov.schutz
        
        # variable schlachtfeld values
        self.TP = self.bTP
        self.wounds = [0,0] #: deep wounds, critical wounds
        self.morale = [12,0,0,0,0] #: base, speech, combat, bias, hero
        self.exp = 0
        self.style = default_style
        
        # even template-soldiers are not all equal
        if template:
            self.morale[0] += rnd(-3,4)
            for i in atscatter:
                self.attack += rnd(-i,i+1)
        
        # schlachtfeld organization values
        self.active = True
        self.alive = True
        self.fled = False
        self.queue = 0
        self.host = None # ToDo: Add Host! Blocker! - Char should get host via Char.sethost after creation (pihalbe) >>> what do you mean? sry, don't get it
	
        return


    def __repr__(self):
        ''' Hopefully nice printout for a soldier.'''
        if self.name != None: 
            scribble = 'Name: ' + self.name
        else: 
            scribble = "Unknown Char"
        scribble += '\nTP: ' + `self.TP`
        scribble += '\nWunden: ' + `self.wounds`
        scribble += '\nAktiv: ' + `self.alive`
        scribble += '\nAngriff: ' + `self.attack` + ', Exp: ' + `self.exp`
        scribble += '\nAktuelle Moral: ' + `self.morale`
        scribble += '\nArt: ' + self.chartype
        scribble += '\nSprache: ' + self.sprache
        if self.host:
            scribble += '\nBatallion: ' + self.host.name
        return scribble


    def sethost(self, host):
        '''Assign the soldier to a hosting Group-Entity.'''
        self.host = host
        return


    def name_me(self,template):
        '''Give the soldier a random name or import it from the file.'''
        if template:
            self.nametype = self.amov.grunddaten['Herkunft']['Sprache']
            self.name = name_it(language=self.nametype)
        else:
            self.name = self.amov.name
        return


    def damage(self, tp, ws, hws):
        '''Damage a character and check for suvivial. Arguments: hitpoints, wounds heavy wounds.'''
        self.TP -= tp + (ws * self.WS[0]) + (hws * self.WS[1]) # self.WS is a list containing the wound value and the heavy wound value of the character. 
        self.wounds[0] += ws
        self.wounds[1] += hws
        self.morale[2] -= ws + 3*hws
        self.checkalive()
        return


    def checkalive(self):
        '''Check if a character is still alive after being damaged.'''
        morale = sum(self.morale)
        # morale effects: for each morale over 15 > negative TP; for every 6 morale over 12 > max hwounds+1; for every 5 morale over 15 > max wounds+1
        self.alive = (self.TP > min(0,15-morale)) and (self.wounds[1] < max(2,morale/6)) and (self.wounds[0] < max(5,morale/3))
        if not self.alive:
            self.active = False
            self.host.active.remove(self)
            print 'removed self: died'
        return


    def checkmorale(self):
        '''
        A morale-check to figure out if the character will cross over or desert his Group.
        This method also automatically determines the fighting style of the soldier.
        '''
        MW = sum(self.host.situation)
        res = ocheck(sum(self.morale),MW)
        host = self.host
        if res < -10 and self.host.enemy:
            self.host.active.remove(self)
            print 'removed self: x over'
            self.sethost(host.enemy)
            self.host.active.append(self)
            self.morale[2] = 0
        elif res < -3:
            self.style = escape_style
        elif res < 0:
            self.style = defensive_style
        elif res < 5:
            self.style = default_style
        else:
            self.style = offensive_style
        self.morale[4] = 0  # reset of hero-added morale
        return


    def escape_check(self):
        '''Check if the character tried to escape and made it alive.'''
        if self.style == escape_style and self.active and self.alive:
            self.active = False
            self.host.active.remove(self)
            print 'removed self: escaped'
            self.sethost(None)
            self.fled = True
        return


    def queue_up(self,amt=-1):
        '''
        Adjust queue-value of character - either reduce by 1 (standard) or raise or lower deliberately.
        Check for activeness.
        '''
        # remove from active-list for queueing
        #print self.name, '\n\nQUEUE_UP\n\n\n', self.host.active
        if amt > 0:
            self.host.active.remove(self)
            print 'removed self: queued'
        
        # adjust queue
        self.queue += amt
        
        # determine activeness
        if self.queue > 0:
            self.active = False
        else:
            self.active = True
            self.host.active.append(self)
        return


    def do_attack(self):
        '''
        Make an attack-roll to be compared to the enemy's roll. Base method to evaluate comabat.
        Contributions: base skill, weapon, armor, +-d6, wounds (deep: each -3, crit: each -6 while in combat), , group situation (strategy & bias), morale bonus
        '''
        tmp = self.attack + self.dam + self.armor + ews() - 3*(self.wounds[0]+2*self.wounds[1]) + self.host.situation[2] - self.host.situation[1] + max(0, (12-sum(self.morale))/3)
        return tmp


    def upgrade(self,expadd,object=('random',0)):
        '''
        Randomly enhance the characters skills and attributes. Dashes are spent and collected until a higher level is reached. Higher attributes and skills are preferred, thus specializing the character.
        expadd: number of dashes (float - subdashes included)
        object: object of enhancement; 'attribute', 'skill' or 'random' + name of object; standard is 'random'
        '''
        return


    def manual_sytle(self, style):
        self.style = styledict[style]
        return




class Leader(Char):
    """A leader of an Army or group. Can emerge from Char().""" 
    def __init__(self,source="default",template=True):
        # additionally import important skills
        if 'Strategie' in self.amov.fertigkeiten:
            self.strategy_skill = self.amov.fertigkeiten['Strategie']['Zahlenwert']
        else:
            self.strategy_skill = 9

        if 'Taktik' in self.amov.fertigkeiten:
            self.tactics_skill = self.amov.fertigkeiten['Taktik']['Zahlenwert']
        else:
            self.tactics_skill = 9

        if 'Rede' in self.amov.fertigkeiten:
            self.speech_skill = self.amov.fertigkeiten['Rede']['Zahlenwert']
        else:
            self.speech_skill = 9

        self.chartype = 'Anführer'
        return


    def __repr__(self):
        '''Print-out modification for leaders.'''
        scribble = Char.__repr__(self)
        scribble += '\nStrategie-Fertigkeit: ' + self.strategy_skill
        scribble += '\nTaktik-Fertigkeit: ' + self.tactics_skill
        scribble += '\nRede-Fertigkeit: ' + self.speech_skill
        return scribble


    def speak(self):
        '''Speak to raise or lower morale of soldiers in his group.'''
        bonus = max(ocheck(self.speech_skill, sum(self.host.situation)) / 3, -6) # capping @ -6 malus
        for i in self.host.active:
            if i.morale[1] < self.speech_skill/3:
                i.morale[1] += bonus
        return


    def plan(self):
        '''Make strategical or tactical considerations to raise or lower attack score of soldiers.'''
        tmp = self.host.grouptype
        if tmp == "Armee":
            skill = self.strategy_skill
        elif tmp =="Gruppe":
            skill = self.tactics_skill
        bonus = ocheck(skill, sum(self.host.situation)) / 3
        self.host.situation[2] = bonus
        return



class Hero(Char):
    """A battle hero. Might be external or emerging from Char() during battle."""
    def __init__(self,host=None,exp=3,weap=4,arm=2,base_tp=24,base_attack=12,morale=18,source=u"tag:1w6.org,2007:Mensch", template=False):
        Char.__init__(self,source,template)
        self.chartype = 'Held'
        return

    def checkmorale(self):
        '''Heroes hail their masters, no morale check!'''
        MW = sum(self.host.situation)
        res = ocheck(sum(self.morale),MW)
        if res < 0:
            self.style = defensive_style
        elif res < 5:
            self.style = default_style
        else:
            self.style = offensive_style
        self.morale[4] = 0  # reset of hero-added morale
        return

    def motivate(self):
        '''Heroes motivate the surrounding fighters.'''
        for i in self.host.active:
            i.morale[4] += self.morale[2]
        self.host.situation[3] += 1
        return


#### CLASSES ####
