#!/bin/env python
# -*- coding: utf-8 -*-

# Schlachtfeld - Großk�mpfe im EWS System 
#   http://rpg-tools-1d6.sf.net
# Copyright © 2007 - 2007 Achim Zien

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
# MA 02110-1301 USA

"""
Schlachtensimulation nach den Regeln des Ein Würfel Systems
"""


#### IMPORTS ####

# for the fighting
from random import randrange as rnd
from random import shuffle
import math

# For selection of armies through command line arguments: 
import sys

# EWS standard dice
from ews import pmw6

# EWS One Roll Fight System
from ews.fight import fight

# Name-Generator
import namen

# Obtain Charakters from yaml-files
from amov import Charakter, Versionsverwaltung
# replaced the alias with the real directory. 

# get the Group and Char classes and related
from group import *
from char import *


#### IMPORTS ####




#### CONTROL PARAMETERS ####

# BIAS

# each factor of biasfactor gives +1 to attack for superior numbers
biasfactor = 1.4

# multiplier for the bias calculation
biasmultiplier = 2.0

# the maximum biasfactor (if multiplied with biasmultiplier, approached asymptotically
biascutoff = 8


## MAYBE STUFF FROM HERE ON DOWNWARDS COULD BE ARMY / RACE DEPENDENT ##

# initial exp gain, to be modified by opponents and own curent exp
expfactor = 1 # disabled

# factor by which the exp of the char goes into diminishing the exp gain
exp_logfactor = 1.5


# POWERRATIO

# weighting of inactive fighters
inactvalue = 0.7

# weighting of wounded fighters
woundvalue = 0.4 # obsolete

# minimum absolute remaining fraction of fighters until resign/flee
powervalue = 0.2

#### CONTROL PARAMETERS ####




#### Top-Level Objects ####

name_objekt = namen.Name()
verbose = True

#### Top-Level Objects ####




#### WRAPPERS ####

def ews():
    """Do one roll with the plusminus d6 die (wrapper for a function in pmw6)."""
    return pmw6.pmw6()

def check(skill,MW):
    """Check if a given skill test reached a min-value (wrapper for a function in pmw6)."""
    return pmw6.check(skill,MW)

def ocheck(skill,MW):
    """Return the degree of success / failure for a skill check (wrapper for a function in pmw6)."""
    return pmw6.ocheck(skill,MW)

#### WRAPPERS ####




#### CLASSES ####

class BattleField:

    def __init__(self,armies):
        '''
        Create a new battle with armies, groups, leaders, soldiers and characters. Consists of several consecutive battle-rounds.
        armies should be a list of combattants (usually 2) containing in a list:
        	1. their army settings - (name,resignratio,fleeratio)
        	2. a list of settings for the contained groups - (name,resignratio,fleeratio)
        	3. a dictionary containing the template-names with a tuple containing template-switch and a list of corresponding tuples (number of soldiers, index number of host(-1 is army),chartype) - {'template_name_1':(True, [(number_count_1a,host#1a,Typus1a),(number_count_1b,host#1b,Typus1b)]), 'template_name_2':(False,[(number_count_2,host#2,Typus2)]), ...}
        '''
	self.output = "" #: The output of the battlefield calculations for use in other programs. 
        if verbose: 
            output = "Pitching the following armies against each other: " +  str(armies)
	    print output
	    self.output += output
        self.Armies = []
        self.Groups = []
        self.Soldiers = []
        self.Leaders = []
        self.Heroes = []
        
        # create multiple armies
        for i in range(len(armies)):
            self.Armies.append(Army(armies[i][0][0],armies[i][0][1],armies[i][0][2]))
            self.Armies[-1].setbattlefield(self)
            
            # create groups within army
            for j in range(len(armies[i][1])):
                self.Groups.append(Group(armies[i][1][j][0],armies[i][1][j][1],armies[i][1][j][2]))
                self.Groups[-1].sethhost(self.Armies[i])
                self.Groups[-1].setbattlefield(self)
                self.Armies[-1].groups.append(self.Groups[-1])
            
            # create soldiers within army
            for j in armies[i][2]:
                
                # detemine soldier type
                if j[3] == 'Soldat':
                    concern = self.Soldiers
                    Model = Char
                elif j[3] == 'Anführer':
                    concern = self.Leaders
                    Model = Leader
                elif j[3] == 'Held':
                    concern = self.Heroes
                    Model = Hero
                else:
                    raise UnmatchedChartype
                
                # spawn soldier
                for k in range(j[1]):
                    spawn = Model(source=j[0],template=j[4])
                    
                    # set the respective host
                    if j[2] == -1:
                        spawn.sethost(self.Armies[-1])
                    else:
                        spawn.sethost(self.Armies[-1].groups[j[2]])
                        
                    # make appropriate entries in soldier & host lists
                    concern.append(spawn)
                    spawn.host.active.append(spawn)
        
        # initialize work-lists of battlefield
        self.Pairs = []
        self.gPairs = []
        self.fGroups = self.Armies + self.Groups
        if verbose:
            print "fGroups:", self.fGroups
        return


    def battle(self):
        '''
        Execute a single round of the battle together with organization of Armies, Groups and Characters.
        Performs the following steps:
            1. Enemy finding for single groups
            2. Characters prepare themselves
            3. Groups prepare themselves
            4. Pairs of groups evaluate their combat modifiers
            5. Pairs of soldiers are formed
            6. They fight each other
        '''
        if verbose:
            print "Armies:", self.Armies
            print "\nAssigning enemies."
        self.auto_enemy()
        if verbose:
            print "\ngPairs:", self.gPairs
            print "\nDoing Char preparations."
        self.char_preparations()
        if verbose:
            print "\nDoing Group preparations."
        self.group_preparations()
        if verbose:
            print "\nBeginning to pair fighters."
        for i in self.gPairs:
            if verbose:
                print "Pairing groups from gPairs"
            self.group_pairings(i)
            if verbose:
                print "pairing chars."
            self.char_pairings(i)
        # if verbose: # Print all fighters
            # print "Finished pairing Groups and fighters."
            # for i in self.Armies: 
                # for j in i.active: 
                    # print j
            # print "\nStarting the fights."
        self.fight_rows()
        return


    def auto_enemy(self):
        '''
        Automatically assign enemies to groups / armies.
        1. shuffle free groups
        2. pick a group
        3. find an appropriate enemy
        4. iterate until no further pairings possible
        '''
        shuffle(self.fGroups)
        if verbose: 
            print "\nCreating pairs from the following groups:", self.fGroups, "\n"
        for i in range(len(self.fGroups)):
            # selfchecks
            if len(self.fGroups) == 0:      #obsolete?
                print "fGroups is empty."   #obsolete?
                break                       #obsolete?
            elif len(self.fGroups) == 1: 
                print "fGroups has only one member."
                break
            else:
                if verbose:
                    print "\nGroup", i
            
            # pick last group & remove it from list
            guy = self.fGroups.pop()
            if verbose: 
                print "Group %d:" % (i,), guy
                
            # stops, when each group has been given a chance to find a partner
            lgroups = len(self.fGroups)
            if i == len(self.fGroups):
                if verbose:
                    print "no groups left to pair with each other."
                break
            
            # checks for presence of soldiers
            # if guy.empty: # guy is always ampty at this stage. It will only be filled when the groups and chars prepare. 
                # print "\nGuy.empty:", guy.empty
                # print "No group to pair: ", guy
                # continue
            
            # tries to find a proper enemy and (if found) removes it from fGroups
            # adds the pair to gPairs
            for j in range(lgroups):
                if verbose:
                    print "fGroups: trying", j, "/", lgroups
                if self.fGroups[j].host != guy.host:
                    if verbose: 
                        print "paired with:", guy.enemy, "."
                    guy.enemy = self.fGroups[j]
                    guy.enemy.enemy = guy
                    self.fGroups = self.fGroups[:j] + self.fGroups[j+1:]
                    self.gPairs.append((guy, guy.enemy))
                    break
        return


    def char_preparations(self):
        '''
        Make all the characters still involved do their preparations:
        Leader(): .plan, .speak and .checkmorale
        Hero(): .motivate and .checkmorale
        Char(): .checkmorale
        '''
        # Leaders
        if verbose: 
            print "Leaders prepare:"
        for i in self.Leaders:
            if i.active:
                if verbose: 
                    print "-", i.name, "plans,", 
                i.plan()
                if verbose: 
                    print "speaks", 
                i.speak()
                if verbose: 
                    print "and seeks strength from the heart."
                i.checkmorale() # leaders get the hero morale bonus 1 round delayed - which is ok for me since they are not within the heap of soldiers >> me too
                
        # Heroes
        if verbose: 
            print "\nHeroes prepare:"
        for i in self.Heroes:
            if i.active:
                if verbose: 
                    print "-", i.name, "motivates", 
                i.motivate()
                if verbose: 
                    print "and seeks strength from the heart."
                i.checkmorale()
                
        # Soldiers
        if verbose: 
            print "\nSoldiers prepare:"
        for i in self.Soldiers:
            if i.active:   # i.alive and not i.queue and not i.fled and i.host
                if verbose: 
                    print "-", i.name, "seeks strength from the heart."
                i.checkmorale()
        return


    def group_preparations(self):
        '''
        Make all non-empty groups do their preparations for battle:
            1. find enemies
            2. count numbers and check for retreat / resignation
        '''
        for i in self.Armies + self.Groups:
            if verbose: 
                print i.name, "counts its fighters,", 
            nact = i.count()
            if verbose: 
                print "finds", nact, "active ones and tries to fight off thoughts of fleeing."
            i.fleeresigncheck()
            #shuffle(i.active) # not consistent with battle lines!
        return


    def group_pairings(self,groups):
        '''Take selected pairs of groups to calculate their powerratio/aliveratio and evaluate their relative bias.'''
        powercheck(groups)
        bias_me(groups)
        return


    def char_pairings(self,groups):
        '''
        Pair the Characters from two Groups to fight each other. Only the Characters from the front line are taken into account.
        groups: should be a tuple of two Group() or Army() instances.
        '''
        self.Pairs = []
        n = min([i.line for i in groups])
        if verbose: 
            for i in groups: 
                print "Lines:", i.line
        if verbose: 
            print "Number of groups:", len(groups)
            for i in range(len(groups)): 
                print "Number of fighters in group", i, ":", len(groups[i].active)
        if verbose:
            print "\nPairs: minimum number of lines: n =", n
        for i in range(n):
            self.Pairs.append([j.active[i] for j in groups]) # ToDo: This sometimes returns a "list index out of range" error.
        return


    def fight_rows(self):
        '''
        Do a single One-on-One fight for every pair of opposing Characters.
        1. compare their skills + dice + modifiers (bias etc.) and get wounded, knocked out etc.
        2. gain experience according to opponents exp, own exp, outer circumstances etc.
        '''
        for i in self.Pairs:
            fight(i)
            experience(i)


    def __repr__(self):
        '''Nice printout of current situation on the BattleField().'''
        scribble = 'Yeah, the proper printout is still lacking'
        return scribble


    def auto_battle(self,infolevel=0):
        '''Automatically executes battles plus intermediate output until the battle ends (flight, extermination, resignation).'''
        # ToDo: implement
        return


    def supplies(self):
        '''Give one of the Armies/Groups additional support in form of new soldiers, leaders, heroes or entire groups.'''
        # ToDo: implement
        # wrapper for Group().method?
        return


#### CLASSES ####




#### FUNCTIONS ####    


def powercheck(Groups):
    """Compare the ratio between fighters who are alive in each group (but not necessarily able to fight).
    Input is a pair of Groups (currently, might be more [how?] lateron).""" 
    for i in Groups:
        i.aliveratio = float(i.nact + i.ninact * inactvalue) / max(1, i.n)
    for i in range(len(Groups)):
        # compare the ratio of fighters who are able to fight on each side. 
        if Groups[1-i].aliveratio != 0: 
            Groups[i].powerratio = Groups[i].aliveratio / Groups[1-i].aliveratio
        # if the other army has no fighters left, just set it to 1. 
        else: 
            Groups[i].powerratio = 1


def bias_me(armies):
    '''Calculate the bias between two groups and store it in Group().situation[1].'''
    if max(armies[0].nact, armies[1].nact) < 5:
        if armies[0].nact > armies[1].nact: 
            bias = int(round(3 * float(armies[0].nact - armies[1].nact)/max(1, armies[1].nact)))
        else: 
            bias = int(round(3 * float(armies[1].nact - armies[0].nact)/max(1, armies[0].nact)))
    else: 
        q = math.log(float(max(1, armies[0].nact))/max(1, armies[1].nact),biasfactor)
        bias = int(round(biasmultiplier * q/(1+abs(q)/biascutoff)))
    armies[0].situation[1] = -bias
    armies[1].situation[1] = bias


#### FUNCTIONS ####

def experience(chars): 
    '''
    Give the characters their respective gained experience according to the weighted Babenhauserheide-formula :_).
    Let them spend their xp if any was gained.
    Check, wheather char wants to retreat.
    '''
    for j in range(2):
        expadd = max(math.log(max(chars[1-j].exp, 1), exp_logfactor), 1) / max(chars[j].exp, 3) # * expfactor
        chars[j].upgrade(expadd)
        chars[j].escape_check()
        if verbose: 
            print '\n', chars[j]
    if verbose: 
        print '\n\n'

#### SELF-CHECK ####

if __name__ == '__main__':
    
    # Army definition: (("Name", ??, ??), [??], [groups])
    # Group definition: ("tag", number_of_soldiers, ??, 'Type_of_chars', from_template?)
    armies = [(('Menschen Armee',0.5,0.4), [],
    [("tag:1w6.org,2007:Mensch",10,-1,'Soldat',True),("tag:draketo.de,2007:Sskreszta",1,-1,"Held", False)]),
        (('Goblin Armee',0.4,0.4), [], [("tag:1w6.org,2007:Goblin",15,-1,'Soldat',True)])]
    
    test = BattleField(armies)
    for i in range(5):
        test.battle()
        for i in test.Armies:
            print i
            print '\n'
    print '... done!'


#### SELF-CHECK ####
