;; Adapted from https://git.dthompson.us/guile-websocket.git/tree/web/socket/server.scm
;;; Copyright © 2015 David Thompson <davet@gnu.org>
;;; Copyright © 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of guile-websocket.
;;;
;;; Guile-websocket is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU Lesser General Public License as
;;; published by the Free Software Foundation; either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Guile-websocket is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Lesser General Public License for more details.
;;;
;;; You should have received a copy of the GNU Lesser General Public
;;; License along with guile-websocket.  If not, see
;;; <http://www.gnu.org/licenses/>.

;;; Commentary:
;;
;; WebSocket server.
;;
;;; Code:

define-module : websocket
    . #:export : run-server websocket-available? make-server-socket

import
    ice-9 match
    rnrs bytevectors
    rnrs io ports
    web request
    web response
    web uri
    fibers
    fibers channels
    fibers internal

; false-if-exception
import
        web socket frame
        web socket utils
define : websocket-available?
    . #t ;; defined? 'write-frame


;; See section 4.2 for explanation of the handshake.
. (define (read-handshake-request client-socket)
  "Read HTTP request from CLIENT-SOCKET that should contain the
headers required for a WebSocket handshake."
  ;; See section 4.2.1.
  (and (not (port-eof? client-socket))
    (read-request client-socket)))

. (define (make-handshake-response client-key)
  "Return an HTTP response object for upgrading to a WebSocket
connection for the client whose key is CLIENT-KEY, a base64 encoded
string."
  ;; See section 4.2.2.
  (if (not client-key)
      (build-response #:code 200)
      (let ((accept-key (make-accept-key (string-trim-both client-key))))
          (build-response #:code 101
              #:headers `((upgrade . ("websocket"))
                          (connection . (upgrade))
                          (sec-websocket-accept . ,accept-key))))))

. (define* (make-server-socket #:key
                             (host #f)
                             (family AF_INET)
                             (addr (if host (inet-pton family host) INADDR_LOOPBACK))
                             (port 8080))
  (let ((sock (socket PF_INET SOCK_STREAM 0)))
    (setsockopt sock SOL_SOCKET SO_REUSEADDR 1)
    (bind sock AF_INET addr port)
    sock))

. (define (accept-new-client server-socket)
  (match (accept server-socket SOCK_NONBLOCK)
    ((client-socket . _) client-socket)))
    
define : write-frame-maybe frame socket
    catch #t
        λ() : write-frame frame socket
        λ(. args)
            format (current-error-port) "Writing to websocket failed ~A\n" args
            . #f

define static-response ;; use with (format #f ... host port)
    . "<html><head><meta charset='utf-8'><title>Dryads Wake</title></head><body><style>input {outline: none; border-width: 0; border-top-width: thin; background-color: #eee;} pre {white-space: pre-wrap;}</style><pre id='dryads-wake-content'></pre><form id='dryads-wake-form'><input type='text' id='dryads-wake-input' autofocus='true' size=42 /></form><script>
      function setupWebsocket () {
        var ws = new WebSocket(`${window.location.protocol === 'https:' ? 'wss' : 'ws'}://${window.location.host}`);
        console.log('initialized websocket');
        ws.onmessage = function(evt) {
            document.getElementById('dryads-wake-content').innerHTML += evt.data;;
            document.getElementById('dryads-wake-input').scrollIntoView();
        };
        ws.onopen = function() {
            console.log('connected');
            setTimeout(() => ws.send(''), 30);
        }
        ws.onclose = function() {
            console.log('closed websocket');
        }
        return ws;
      }
      var ws = setupWebsocket();
      var form = document.getElementById('dryads-wake-form');
      var inp = document.getElementById('dryads-wake-input');
      form.onsubmit = () => { ws.send(inp.value); inp.value = ''; return false };
    </script></body></html>"

. (define (serve-client client-socket channel)
  "Serve client connected via CLIENT-SOCKET by performing the HTTP
handshake and listening for control and data frames.  HANDLER is
called for each complete message that is received."
  ;; create delimited contituation to get a co-routine where the handler can jump back here when it needs input.
  (define (show-symbol x)
    (map write-char (string->list (symbol->string x))))
    ;; #f)
  (define (handle-data-frame type data)
    (show-symbol 'handling-data-frame)
    (sleep 0.001)
    (let ((message 
      (match type
        ('text   (let ((msg (utf8->string data)))
            ;; if the message is not empty, add a linebreak (that’s swallowed by the websocket)
            (if (string-null? msg) msg (string-append msg "\n"))))
        ('binary data))))
       (show-symbol 'put-websocket-message)
       (when (not ready) (set! ready #t) (put-message channel 'first-data-frame-handled))
       (sleep 0)
       (put-message channel message))
   (show-symbol 'handled-data-frame))

  (define (read-frame-maybe)
    (catch #t
        (λ() (and (not (port-eof? client-socket))
          (read-frame client-socket)))
        (λ (. args) 
          (format (current-error-port) "Reading from websocket failed ~A\n" args)
          #f)))
         
  (define ready #f)

  ;; Perform the HTTP handshake and upgrade to WebSocket protocol.
  (define request (read-handshake-request client-socket))
  (when request
   (let* ((client-key (assoc-ref (request-headers request) 'sec-websocket-key))
          (response (make-handshake-response client-key)))
    (when (false-if-exception (write-response response client-socket))
     (if (not client-key)
      (begin
          (catch #t
            (λ() (let ((host (assoc-ref (request-headers request) 'host))
                       (ws "wss"))
                   (map (λ(x) (write-char x client-socket))
                      (string->list static-response))))
            (λ(. args) #f))
          (close client-socket)
          (put-message channel #f))
      (begin
       (let loop ((fragments '())
                 (type #f))
        (show-symbol 'reading)
        (sleep 0.001)
        
        (let ((frame (read-frame-maybe)))
          (show-symbol 'read)
          (cond
           ;; EOF - port is closed.
           ((not frame)
            (show-symbol 'not-frame)
            (close-port client-socket))
           ;; Per section 5.4, control frames may appear interspersed
           ;; along with a fragmented message.
           ((close-frame? frame)
            (show-symbol 'close-frame)
            ;; Per section 5.5.1, echo the close frame back to the
            ;; client before closing the socket.  The client may no
            ;; longer be listening.
            (false-if-exception
             (write-frame-maybe (make-close-frame (frame-data frame)) client-socket))
            (close-port client-socket))
           ((ping-frame? frame)
            (show-symbol 'ping-frame)
            ;; Per section 5.5.3, a pong frame must include the exact
            ;; same data as the ping frame.
            (write-frame-maybe (make-pong-frame (frame-data frame)) client-socket)
            (loop fragments type))
           ((pong-frame? frame) ; silently ignore pongs
            (show-symbol 'pong-frame)
            (loop fragments type))
           ((first-fragment-frame? frame) ; begin accumulating fragments
            (show-symbol 'first-fragment-frame)
            (loop (list frame) (frame-type frame)))
           ((final-fragment-frame? frame) ; concatenate all fragments
            (show-symbol 'final-fragment-frame)
            (handle-data-frame type (frame-concatenate (reverse fragments)))
            (loop '() #f))
           ((fragment-frame? frame) ; add a fragment
            (show-symbol 'fragment-frame)
            (loop (cons frame fragments) type))
           ((data-frame? frame) ; unfragmented data frame
            (show-symbol 'data-frame)
            (handle-data-frame (frame-type frame) (frame-data frame))
            (loop '() #f)))))))))))


define* : run-server handler #:optional (server-socket (make-server-socket))
    . "Run WebSocket server on SERVER-SOCKET.  HANDLER, a procedure that
accepts a single argument, is called for each complete message that
the server receives from a client.  When the message is in text
format, HANDLER is passed a string.  When the message is in binary
format, HANDLER is passed a bytevector.  HANDLER must return either a
string, bytevector, or #f.  Strings and bytevectors are sent to the
client in response to their message, and #f indicates that nothing
should be sent back."
    listen server-socket 1
    fcntl server-socket F_SETFL (logior O_NONBLOCK (fcntl server-socket F_GETFL))
    sigaction SIGPIPE SIG_IGN
    run-fibers
        λ :
          let loop :
            define client-socket : accept-new-client server-socket
            if : not client-socket
              sleep 0.001
              catch #t
               λ :
                 let : : input-channel : make-channel
                   spawn-fiber
                       λ ()
                           serve-client client-socket input-channel
                       . #:parallel? #t
                   ;; if we open a websocket, run the game handler
                   when : get-message input-channel
                     spawn-fiber
                         λ ()
                             handler client-socket write-frame-maybe make-text-frame input-channel 
                         . #:parallel? #t
               λ : . args
                 . #f
            loop
