;;; babcore.el --- Core Emacs Customizations everyone should have

;; Copyright (C) 2017 Arne Babenhauserheide

;; Author: Arne Babenhauserheide (and various others in Emacswiki and elsewhere).
;; Maintainer: Arne Babenhauserheide
;; Created 29. August 2017
;; Version: 0.4.0
;; Keywords: convenience internal abbrev
;; Homepage: https://bitbucket.org/ArneBab/babcore.el/

;; Package-Requires: ((use-package "0") (use-package-chords "0") (goto-last-change "0") (key-chord "0") (fic-mode "0") (legalese "0") (auto-complete "0") (saveplace "0"))

;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 3
;; of the License, or (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program. If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:
;; Quick Start / installation:
;; 1. Download this file and put it next to other files Emacs includes
;; 2. Add this to you .emacs file and restart emacs:
;; 
;;      (require 'babcore)
;;
;; Use Case: Use a common core configuration so you can avoid the
;;   tedious act of gathering all the basic stuff over the years and
;;   can instead concentrate on the really cool new stuff Emacs offers
;;   you.
;;

;;; Change Log:

;; 2017-08-29 - Initial release
;; 2018-05-02 - more fic-mode

;;; Code:

;; get use-package, see https://github.com/jwiegley/use-package
(unless (package-installed-p 'use-package)
  (package-refresh-contents)
  (package-install 'use-package))

(eval-when-compile
  (require 'use-package))
(require 'use-package)

; go to the last change
(use-package goto-last-change
  :ensure t
  :bind (([(control .)] . goto-last-change)
         ;; M-. can conflict with etags tag search. But C-. can get overwritten
         ;; by flyspell-auto-correct-word. And goto-last-change needs a really
         ;; fast key.
         ([(meta .)] . goto-last-change)
         ;; ensure that even in worst case some goto-last-change is available
         ([(control meta .)] . goto-last-change)))

; use key chords invoke commands
(use-package use-package-chords
  :ensure t
  :config (progn
           (key-chord-mode 1)
           ;; buffer actions
           (key-chord-define-global "vb"     'eval-buffer)
           ;; frame actions
           (key-chord-define-global "xö"     'other-window)
           (key-chord-define-global "xc"     'other-window)
           (key-chord-define-global "x1"     'delete-other-windows)
           (key-chord-define-global "x0"     'delete-window)
           ;; file actions
           (key-chord-define-global "xs"     'save-buffer)
           (defun babcore-kill-this-buffer-if-not-modified ()
             (interactive)
             ;; taken from menu-bar.el
             (if (menu-bar-non-minibuffer-window-p)
                 (kill-buffer-if-not-modified (current-buffer))
               (abort-recursive-edit)))
           ;; file actions
           (key-chord-define-global "bf"     'ido-switch-buffer)
           (key-chord-define-global "bk"     'babcore-kill-this-buffer-if-not-modified)
           (key-chord-define-global "cf"     'ido-find-file)
           (key-chord-define-global "vc"     'vc-next-action)))

(setq visible-bell t)

; save the current macro as reusable function.
(defun save-current-kbd-macro-to-dot-emacs (name)
  "Save the current macro as named function definition inside
your initialization file so you can reuse it anytime in the
future."
  (interactive "SSave Macro as: ")
  (name-last-kbd-macro name)
  (save-excursion 
    (find-file-literally user-init-file)
    (goto-char (point-max))
    (insert "\n\n;; Saved macro\n")
    (insert-kbd-macro name)
    (insert "\n")))

; Activate transparent GnuPG encryption.
(use-package epa-file ;; no need to ensure, since it is part of Emacs
  :config (epa-file-enable))

;; Highlight TODO and FIXME in comments 
(use-package fic-mode
  :ensure t
  :config (progn
            (defun add-something-to-mode-hooks (mode-list something)
              "helper function to add a callback to multiple hooks"
              (dolist (mode mode-list)
                (add-hook (intern (concat (symbol-name mode) "-mode-hook")) something)))

            (add-something-to-mode-hooks '(prog text markdown latex org) 'fic-mode))) ;; prog captures all programming modes

; save the place in files
(use-package saveplace
  :ensure t
  :config (setq-default save-place t))

;; Inline auto completion and suggestions
(defcustom babcore-completion-package 'auto-complete
  "Select the completion package."
  :type 'symbol
  :options '(auto-complete)) ;; TODO: add company
(if (equal 'auto-complete babcore-completion-package)
    (use-package auto-complete
                 :ensure t
                 :config
                 ;; avoid competing with org-mode templates.
                 (add-hook 'org-mode-hook
                           (lambda ()
                             (require 'org)
                             (make-local-variable 'ac-stop-words)
                             (loop for template in org-structure-template-alist do
                                   (add-to-list 'ac-stop-words 
                                                (concat "<" (car template)))))))
  ;; TODO: add company
  )

(use-package legalese :ensure t)

;;; Open files as root - quickly
(defcustom find-file-root-prefix "/sudo:root@localhost:"
"Tramp root prefix to use.")

(defun find-file-as-root ()
  "Like `ido-find-file, but automatically edit the file with
root-privileges (using tramp/sudo), if the file is not writable by
user."
  (interactive)
  (let ((file (ido-read-file-name "Edit as root: ")))
    (unless (file-writable-p file)
      (setq file (concat find-file-root-prefix file)))
    (find-file file)))
;; or some other keybinding...
;; (global-set-key (kbd "C-x F") 'djcb-find-file-as-root)

(defun find-current-as-root ()
  "Reopen current file as root"
  (interactive)
  (set-visited-file-name (concat find-file-root-prefix (buffer-file-name)))
  (setq buffer-read-only nil))

(defun babcore-show-frame (&optional frame)
  "Show the current Emacs frame or the FRAME given as argument.

And make sure that it really shows up!"
  (raise-frame)
  ; yes, you have to call this twice. Don’t ask me why…
  ; select-frame-set-input-focus calls x-focus-frame and does a bit of
  ; additional magic.
  (select-frame-set-input-focus (selected-frame))
  (select-frame-set-input-focus (selected-frame)))

;; let emacs blink when something interesting happens.
;; in KDE this marks the active Emacs icon in the tray.
(defun babcore-x-urgency-hint (frame arg &optional source)
  "Set the x-urgency hint for the FRAME to ARG: 

- If arg is nil, unset the urgency.
- If arg is any other value, set the urgency.

SOURCE can be used to send the type of the client.

If you unset the urgency, you still have to visit the frame to make the urgency setting disappear (at least in KDE)."
    (let* ((wm-hints (append (x-window-property 
                "WM_HINTS" frame "WM_HINTS" source nil t) nil))
     (flags (car wm-hints)))
    (setcar wm-hints
        (if arg
        (logior flags #x100)
          (logand flags (lognot #x100))))
    (x-change-window-property "WM_HINTS" wm-hints frame "WM_HINTS" 32 t)))

(defun babcore-x-urgent (&optional arg)
  "Mark the current Emacs frame as requiring urgent attention.

With a prefix argument (ARG) which does not equal a boolean value of nil, remove the urgency flag (which might or might not change display, depending on the window manager)."
  (interactive "P")
  (let (frame (selected-frame))
  (babcore-x-urgency-hint frame (not arg))))

; Default KDE keybindings to make emacs nicer integrated into KDE.

; can treat C-m as its own mapping.
; (define-key input-decode-map "\C-m" [?\C-1])

(defun babcore-revert-buffer-preserve-modes ()
  "Revert the current buffer while keeping the modes."
  (interactive)
  (revert-buffer t nil t))

; C-m shows/hides the menu bar - thanks to http://stackoverflow.com/questions/2298811/how-to-turn-off-alternative-enter-with-ctrlm-in-linux
; f5 reloads
(defconst babcore-default-keys-minor-mode-map
  (let ((map (make-sparse-keymap)))
    (set-keymap-parent map text-mode-map)
    (define-key map [f5] 'babcore-revert-buffer-preserve-modes)
    (define-key map [?\C-1] 'menu-bar-mode)
    (define-key map [?\C-+] 'text-scale-increase)
    (define-key map [?\C--] 'text-scale-decrease) ; shadows 'negative-argument which is also available via M-- and C-M--, though.
    (define-key map [C-kp-add] 'text-scale-increase)
    (define-key map [C-kp-subtract] 'text-scale-decrease)
    map)
  "Keymap for command ‘babcore-default-keys-minor-mode’.")

;; Minor mode for keypad control
(define-minor-mode babcore-default-keys-minor-mode
  "Adds some default KDE keybindings"
  :global t
  :require 'babcore ;; required for global minor modes
  :init-value t
  :lighter ""
  :keymap 'babcore-default-keys-minor-mode-map
  )

;; Set the frame title as by http://www.emacswiki.org/emacs/FrameTitle
(setq frame-title-format (list "%b ☺ " (user-login-name) "@" (system-name) "%[ - GNU %F " emacs-version)
      icon-title-format (list "%b ☻ " (user-login-name) "@" (system-name) " - GNU %F " emacs-version))

;;;;;;;;;;;;;
;;; Fixes ;;;
;;;;;;;;;;;;;

;; Make comint recognize passwords in virtually all languages.
(defcustom comint-password-prompt-regexp
  (concat
   "\\(^ *\\|"
   (regexp-opt
    '("Enter" "enter" "Enter same" "enter same" "Enter the" "enter the"
      "Old" "old" "New" "new" "'s" "login"
      "Kerberos" "CVS" "UNIX" " SMB" "LDAP" "[sudo]" "Repeat" "Bad") t)
   " +\\)"
   (regexp-opt
    '("Adgangskode" "adgangskode" "Contrasenya" "contrasenya" "Contraseña" "contraseña" "Geslo" "geslo" "Hasło" "hasło" "Heslo" "heslo" "Iphasiwedi" "iphasiwedi" "Jelszó" "jelszó" "Lozinka" "lozinka" "Lösenord" "lösenord" "Mot de passe " "Mot de Passe " "mot de Passe " "mot de passe " "Mật khẩu " "mật khẩu" "Parola" "parola" "Pasahitza" "pasahitza" "Pass phrase" "pass Phrase" "pass phrase" "Passord" "passord" "Passphrase" "passphrase" "Password" "password" "Passwort" "passwort" "Pasvorto" "pasvorto" "Response" "response" "Salasana" "salasana" "Senha" "senha" "Wachtwoord" "wachtwoord" "slaptažodis" "slaptažodis" "Лозинка" "лозинка" "Пароль" "пароль" "ססמה" "كلمة السر" "गुप्तशब्द" "शब्दकूट" "গুপ্তশব্দ" "পাসওয়ার্ড" "ਪਾਸਵਰਡ" "પાસવર્ડ" "ପ୍ରବେଶ ସଙ୍କେତ" "கடவுச்சொல்" "సంకేతపదము" "ಗುಪ್ತಪದ" "അടയാളവാക്ക്" "රහස්පදය" "ពាក្យសម្ងាត់ ៖ " "パスワード" "密码" "密碼" "암호"))
   "\\(?:\\(?:, try\\)? *again\\| (empty for no passphrase)\\| (again)\\)?\
\\(?: for [^:]+\\)?:\\s *\\'")
  "Regexp matching prompts for passwords in the inferior process.
This is used by `comint-watch-for-password-prompt'."
  :version "24.3"
  :type 'regexp
  :group 'comint)

;; Mark all AC_* and AS_* functions as builtin.
(add-hook 'autoconf-mode-hook
          (lambda ()
            (add-to-list 'autoconf-font-lock-keywords '("\\(\\(AC\\|AS\\|AM\\)_.+?\\)\\((\\|\n\\)" (1 font-lock-builtin-face)))))

; tell emacs to ignore alt-gr clicks needed for M4 in the Neo Layout.
(define-key special-event-map (kbd "<key-17>") 'ignore)
(define-key special-event-map (kbd "<M-key-17>") 'ignore)

; yank-pop should yank if the last command was no yank.
(defun yank-pop (&optional arg)
  "Replace just-yanked stretch of killed text with a different stretch.
At such a time, the region contains a stretch of reinserted
previously-killed text.  `yank-pop' deletes that text and inserts in its
place a different stretch of killed text.

When ARG is nil, the previous kill is inserted.
With ARG N, insert the Nth previous kill.
If N is negative, this is a more recent kill.

The sequence of kills wraps around, so that after the oldest one
comes the newest one.

When this command inserts killed text into the buffer, it honors
`yank-excluded-properties' and `yank-handler' as described in the
doc string for `insert-for-yank-1', which see."
  (interactive "*p")
  (if (not (eq last-command 'yank))
      (yank)
    (setq this-command 'yank)
    (unless arg (setq arg 1))
    (let ((inhibit-read-only t)
          (before (< (point) (mark t))))
      (if before
          (funcall (or yank-undo-function 'delete-region) (point) (mark t))
        (funcall (or yank-undo-function 'delete-region) (mark t) (point)))
      (setq yank-undo-function nil)
      (set-marker (mark-marker) (point) (current-buffer))
      (insert-for-yank (current-kill arg))
      ;; Set the window start back where it was in the yank command,
      ;; if possible.
      (set-window-start (selected-window) yank-window-start t)
      (if before
          ;; This is like exchange-point-and-mark, but doesn't activate the mark.
          ;; It is cleaner to avoid activation, even though the command
          ;; loop would deactivate the mark because we inserted text.
          (goto-char (prog1 (mark t)
                       (set-marker (mark-marker) (point) (current-buffer))))))
    nil))

;; colorful compilation buffer
(use-package ansi-color
  :ensure t
  :config 
  (progn
        (defun babcore-colorize-compilation-buffer ()
          "show shell colors in the compilation buffer"
          (let ((inhibit-read-only t))
            (ansi-color-apply-on-region compilation-filter-start (point))))
        (add-hook 'compilation-filter-hook 'babcore-colorize-compilation-buffer)))

(provide 'babcore)
;;; babcore.el ends here
